
/* Chrysalide - Outil d'analyse de fichiers binaires
 * descriptions.h - définition des descriptions des instructions ARMv7
 *
 * Copyright (C) 2019 Cyrille Bagard
 *
 *  This file is part of Chrysalide.
 *
 *  Chrysalide is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Chrysalide is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Chrysalide.  If not, see <http://www.gnu.org/licenses/>.
 */


#ifndef _PLUGINS_ARM_V7_OPCODES_DESCRIPTIONS_H
#define _PLUGINS_ARM_V7_OPCODES_DESCRIPTIONS_H


#include "identifiers.h"



/* Enumération de tous les mots clefs */
static char *_armv7_descriptions[AOP7_COUNT] = {

    [AOP7_A8_ADC_IMMEDIATE] = "Add with Carry (immediate) adds an immediate value and the Carry flag value to a register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_ADC_REGISTER] = "Add with Carry (register) adds a register value, the Carry flag value, and an optionally-shifted register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_ADC_REGISTER_SHIFTED_REGISTER] = "Add with Carry (register-shifted register) adds a register value, the Carry flag value, and a register-shifted register value. It writes the result to the destination register, and can optionally update the condition flags based on the result.",
    [AOP7_A8_ADD_IMMEDIATE_THUMB] = "This instruction adds an immediate value to a register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_ADD_IMMEDIATE_ARM] = "This instruction adds an immediate value to a register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_ADD_REGISTER_THUMB] = "This instruction adds a register value and an optionally-shifted register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_ADD_REGISTER_ARM] = "This instruction adds a register value and an optionally-shifted register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_ADD_REGISTER_SHIFTED_REGISTER] = "Add (register-shifted register) adds a register value and a register-shifted register value. It writes the result to the destination register, and can optionally update the condition flags based on the result.",
    [AOP7_A8_ADD_SP_PLUS_IMMEDIATE] = "This instruction adds an immediate value to the SP value, and writes the result to the destination register.",
    [AOP7_A8_ADD_SP_PLUS_REGISTER_THUMB] = "This instruction adds an optionally-shifted register value to the SP value, and writes the result to the destination register.",
    [AOP7_A8_ADD_SP_PLUS_REGISTER_ARM] = "This instruction adds an optionally-shifted register value to the SP value, and writes the result to the destination register.",
    [AOP7_A8_ADR] = "This instruction adds an immediate value to the PC value to form a PC-relative address, and writes the result to the destination register.",
    [AOP7_A8_AND_IMMEDIATE] = "This instruction performs a bitwise AND of a register value and an immediate value, and writes the result to the destination register.",
    [AOP7_A8_AND_REGISTER] = "This instruction performs a bitwise AND of a register value and an optionally-shifted register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_AND_REGISTER_SHIFTED_REGISTER] = "This instruction performs a bitwise AND of a register value and a register-shifted register value. It writes the result to the destination register, and can optionally update the condition flags based on the result.",
    [AOP7_A8_ASR_IMMEDIATE] = "Arithmetic Shift Right (immediate) shifts a register value right by an immediate number of bits, shifting in copies of its sign bit, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_ASR_REGISTER] = "Arithmetic Shift Right (register) shifts a register value right by a variable number of bits, shifting in copies of its sign bit, and writes the result to the destination register. The variable number of bits is read from the bottom byte of a register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_B] = "Branch causes a branch to a target address.",
    [AOP7_A8_BFC] = "Bit Field Clear clears any number of adjacent bits at any position in a register, without affecting the other bits in the register.",
    [AOP7_A8_BFI] = "Bit Field Insert copies any number of low order bits from a register into the same number of adjacent bits at any position in the destination register.",
    [AOP7_A8_BIC_IMMEDIATE] = "Bitwise Bit Clear (immediate) performs a bitwise AND of a register value and the complement of an immediate value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_BIC_REGISTER] = "Bitwise Bit Clear (register) performs a bitwise AND of a register value and the complement of an optionally-shifted register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_BIC_REGISTER_SHIFTED_REGISTER] = "Bitwise Bit Clear (register-shifted register) performs a bitwise AND of a register value and the complement of a register-shifted register value. It writes the result to the destination register, and can optionally update the condition flags based on the result.",
    [AOP7_A8_BKPT] = "Breakpoint causes a software breakpoint to occur. Breakpoint is always unconditional, even when inside an IT block.",
    [AOP7_A8_BL_BLX_IMMEDIATE] = "Branch with Link calls a subroutine at a PC-relative address. Branch with Link and Exchange Instruction Sets (immediate) calls a subroutine at a PC-relative address, and changes instruction set from ARM to Thumb, or from Thumb to ARM.",
    [AOP7_A8_BLX_REGISTER] = "Branch with Link and Exchange (register) calls a subroutine at an address and instruction set specified by a register.",
    [AOP7_A8_BX] = "Branch and Exchange causes a branch to an address and instruction set specified by a register.",
    [AOP7_A8_BXJ] = "Branch and Exchange Jazelle attempts to change to Jazelle state. If the attempt fails, it branches to an address and instruction set specified by a register as though it were a BX instruction. In an implementation that includes the Virtualization Extensions, if HSTR.TJDBX is set to 1, execution of a BXJ instruction in a Non-secure mode other than Hyp mode generates a Hyp Trap exception. For more information see Trapping accesses to Jazelle functionality on page B1-1255.",
    [AOP7_A8_CBNZ_CBZ] = "Compare and Branch on Nonzero and Compare and Branch on Zero compare the value in a register with zero, and conditionally branch forward a constant value. They do not affect the condition flags.",
    [AOP7_A8_CDP_CDP2] = "Coprocessor Data Processing tells a coprocessor to perform an operation that is independent of ARM core registers and memory. If no coprocessor can execute the instruction, an Undefined Instruction exception is generated. This is a generic coprocessor instruction. Some of the fields have no functionality defined by the architecture and are free for use by the coprocessor instruction set designer. These are the opc1, opc2, CRd, CRn, and CRm fields. However, coprocessors CP8-CP15 are reserved for use by ARM, and this manual defines the valid CDP and CDP2 instructions when coproc is in the range p8-p15. For more information see Coprocessor support on page A2-94.",
    [AOP7_A8_CLREX] = "Clear-Exclusive clears the local record of the executing processor that an address has had a request for an exclusive access.",
    [AOP7_A8_CLZ] = "Count Leading Zeros returns the number of binary zero bits before the first binary one bit in a value.",
    [AOP7_A8_CMN_IMMEDIATE] = "Compare Negative (immediate) adds a register value and an immediate value. It updates the condition flags based on the result, and discards the result.",
    [AOP7_A8_CMN_REGISTER] = "Compare Negative (register) adds a register value and an optionally-shifted register value. It updates the condition flags based on the result, and discards the result.",
    [AOP7_A8_CMN_REGISTER_SHIFTED_REGISTER] = "Compare Negative (register-shifted register) adds a register value and a register-shifted register value. It updates the condition flags based on the result, and discards the result.",
    [AOP7_A8_CMP_IMMEDIATE] = "Compare (immediate) subtracts an immediate value from a register value. It updates the condition flags based on the result, and discards the result.",
    [AOP7_A8_CMP_REGISTER] = "Compare (register) subtracts an optionally-shifted register value from a register value. It updates the condition flags based on the result, and discards the result.",
    [AOP7_A8_CMP_REGISTER_SHIFTED_REGISTER] = "Compare (register-shifted register) subtracts a register-shifted register value from a register value. It updates the condition flags based on the result, and discards the result.",
    [AOP7_A8_DBG] = "Debug Hint provides a hint to debug and related systems. See their documentation for what use (if any) they make of this instruction.",
    [AOP7_A8_DMB] = "Data Memory Barrier is a memory barrier that ensures the ordering of observations of memory accesses, see Data Memory Barrier (DMB) on page A3-151.",
    [AOP7_A8_DSB] = "Data Synchronization Barrier is a memory barrier that ensures the completion of memory accesses, see Data Synchronization Barrier (DSB) on page A3-152.",
    [AOP7_A8_EOR_IMMEDIATE] = "Bitwise Exclusive OR (immediate) performs a bitwise Exclusive OR of a register value and an immediate value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_EOR_REGISTER] = "Bitwise Exclusive OR (register) performs a bitwise Exclusive OR of a register value and an optionally-shifted register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_EOR_REGISTER_SHIFTED_REGISTER] = "Bitwise Exclusive OR (register-shifted register) performs a bitwise Exclusive OR of a register value and a register-shifted register value. It writes the result to the destination register, and can optionally update the condition flags based on the result.",
    [AOP7_A8_ISB] = "Instruction Synchronization Barrier flushes the pipeline in the processor, so that all instructions following the ISB are fetched from cache or memory, after the instruction has been completed. It ensures that the effects of context changing operations executed before the ISB instruction are visible to the instructions fetched after the ISB. Context changing operations include changing the Address Space Identifier (ASID), TLB maintenance operations, branch predictor maintenance operations, and all changes to the CP15 registers. In addition, any branches that appear in program order after the ISB instruction are written into the branch prediction logic with the context that is visible after the ISB instruction. This is needed to ensure correct execution of the instruction stream.",
    [AOP7_A8_IT] = "If-Then makes up to four following instructions (the IT block) conditional. The conditions for the instructions in the IT block are the same as, or the inverse of, the condition the IT instruction specifies for the first instruction in the block. The IT instruction itself does not affect the condition flags, but the execution of the instructions in the IT block can change the condition flags. 16-bit instructions in the IT block, other than CMP, CMN and TST, do not set the condition flags. An IT instruction with the AL condition can be used to get this changed behavior without conditional execution. The architecture permits exception return to an instruction in the IT block only if the restoration of the CPSR restores ITSTATE to a state consistent with the conditions specified by the IT instruction. Any other exception return to an instruction in an IT block is UNPREDICTABLE. Any branch to a target instruction in an IT block is not permitted, and if such a branch is made it is UNPREDICTABLE what condition is used when executing that target instruction and any subsequent instruction in the IT block. See also Conditional instructions on page A4-162 and Conditional execution on page A8-288.",
    [AOP7_A8_LDC_LDC2_IMMEDIATE] = "Load Coprocessor loads memory data from a sequence of consecutive memory addresses to a coprocessor. If no coprocessor can execute the instruction, an Undefined Instruction exception is generated. This is a generic coprocessor instruction. Some of the fields have no functionality defined by the architecture and are free for use by the coprocessor instruction set designer. These are the D bit, the CRd field, and in the Unindexed addressing mode only, the imm8 field. However, coprocessors CP8-CP15 are reserved for use by ARM, and this manual defines the valid LDC and LDC2 instructions when coproc is in the range p8-p15. For more information see Coprocessor support on page A2-94. In an implementation that includes the Virtualization Extensions, the permitted LDC access to a system control register can be trapped to Hyp mode, meaning that an attempt to execute an LDC instruction in a Non-secure mode other than Hyp mode, that would be permitted in the absence of the Hyp trap controls, generates a Hyp Trap exception. For more information, see Trapping general CP14 accesses to debug registers on page B1-1260. Note For simplicity, the LDC pseudocode does not show this possible trap to Hyp mode.",
    [AOP7_A8_LDC_LDC2_LITERAL] = "Load Coprocessor loads memory data from a sequence of consecutive memory addresses to a coprocessor. If no coprocessor can execute the instruction, an Undefined Instruction exception is generated. This is a generic coprocessor instruction. Some of the fields have no functionality defined by the architecture and are free for use by the coprocessor instruction set designer. These are the D bit, the CRd field, and in the Unindexed addressing mode only, the imm8 field. However, coprocessors CP8-CP15 are reserved for use by ARM, and this manual defines the valid LDC and LDC2 instructions when coproc is in the range p8-p15. For more information see Coprocessor support on page A2-94. In an implementation that includes the Virtualization Extensions, the permitted LDC access to a system control register can be trapped to Hyp mode, meaning that an attempt to execute an LDC instruction in a Non-secure mode other than Hyp mode, that would be permitted in the absence of the Hyp trap controls, generates a Hyp Trap exception. For more information, see Trapping general CP14 accesses to debug registers on page B1-1260. Note For simplicity, the LDC pseudocode does not show this possible trap to Hyp mode.",
    [AOP7_A8_LDM_LDMIA_LDMFD_THUMB] = "Load Multiple Increment After (Load Multiple Full Descending) loads multiple registers from consecutive memory locations using an address from a base register. The consecutive memory locations start at this address, and the address just above the highest of those locations can optionally be written back to the base register. The registers loaded can include the PC, causing a branch to a loaded address. Related system instructions are LDM (User registers) on page B9-1986 and LDM (exception return) on page B9-1984.",
    [AOP7_A8_LDM_LDMIA_LDMFD_ARM] = "Load Multiple Increment After (Load Multiple Full Descending) loads multiple registers from consecutive memory locations using an address from a base register. The consecutive memory locations start at this address, and the address just above the highest of those locations can optionally be written back to the base register. The registers loaded can include the PC, causing a branch to a loaded address. Related system instructions are LDM (User registers) on page B9-1986 and LDM (exception return) on page B9-1984.",
    [AOP7_A8_LDMDA_LDMFA] = "Load Multiple Decrement After (Load Multiple Full Ascending) loads multiple registers from consecutive memory locations using an address from a base register. The consecutive memory locations end at this address, and the address just below the lowest of those locations can optionally be written back to the base register. The registers loaded can include the PC, causing a branch to a loaded address. Related system instructions are LDM (User registers) on page B9-1986 and LDM (exception return) on page B9-1984.",
    [AOP7_A8_LDMDB_LDMEA] = "Load Multiple Decrement Before (Load Multiple Empty Ascending) loads multiple registers from consecutive memory locations using an address from a base register. The consecutive memory locations end just below this address, and the address of the lowest of those locations can optionally be written back to the base register. The registers loaded can include the PC, causing a branch to a loaded address. Related system instructions are LDM (User registers) on page B9-1986 and LDM (exception return) on page B9-1984.",
    [AOP7_A8_LDMIB_LDMED] = "Load Multiple Increment Before (Load Multiple Empty Descending) loads multiple registers from consecutive memory locations using an address from a base register. The consecutive memory locations start just above this address, and the address of the last of those locations can optionally be written back to the base register. The registers loaded can include the PC, causing a branch to a loaded address. Related system instructions are LDM (User registers) on page B9-1986 and LDM (exception return) on page B9-1984.",
    [AOP7_A8_LDR_IMMEDIATE_THUMB] = "Load Register (immediate) calculates an address from a base register value and an immediate offset, loads a word from memory, and writes it to a register. It can use offset, post-indexed, or pre-indexed addressing. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDR_IMMEDIATE_ARM] = "Load Register (immediate) calculates an address from a base register value and an immediate offset, loads a word from memory, and writes it to a register. It can use offset, post-indexed, or pre-indexed addressing. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDR_LITERAL] = "Load Register (literal) calculates an address from the PC value and an immediate offset, loads a word from memory, and writes it to a register. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDR_REGISTER_THUMB] = "Load Register (register) calculates an address from a base register value and an offset register value, loads a word from memory, and writes it to a register. The offset register value can optionally be shifted. For information about memory accesses, see Memory accesses on page A8-294. The Thumb form of LDR (register) does not support register writeback.",
    [AOP7_A8_LDR_REGISTER_ARM] = "Load Register (register) calculates an address from a base register value and an offset register value, loads a word from memory, and writes it to a register. The offset register value can optionally be shifted. For information about memory accesses, see Memory accesses on page A8-294.",
    [AOP7_A8_LDRB_IMMEDIATE_THUMB] = "Load Register Byte (immediate) calculates an address from a base register value and an immediate offset, loads a byte from memory, zero-extends it to form a 32-bit word, and writes it to a register. It can use offset, post-indexed, or pre-indexed addressing. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDRB_IMMEDIATE_ARM] = "Load Register Byte (immediate) calculates an address from a base register value and an immediate offset, loads a byte from memory, zero-extends it to form a 32-bit word, and writes it to a register. It can use offset, post-indexed, or pre-indexed addressing. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDRB_LITERAL] = "Load Register Byte (literal) calculates an address from the PC value and an immediate offset, loads a byte from memory, zero-extends it to form a 32-bit word, and writes it to a register. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDRB_REGISTER] = "Load Register Byte (register) calculates an address from a base register value and an offset register value, loads a byte from memory, zero-extends it to form a 32-bit word, and writes it to a register. The offset register value can optionally be shifted. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDRBT] = "Load Register Byte Unprivileged loads a byte from memory, zero-extends it to form a 32-bit word, and writes it to a register. For information about memory accesses see Memory accesses on page A8-294. The memory access is restricted as if the processor were running in User mode. This makes no difference if the processor is actually running in User mode. LDRBT is UNPREDICTABLE in Hyp mode. The Thumb instruction uses an offset addressing mode, that calculates the address used for the memory access from a base register value and an immediate offset, and leaves the base register unchanged. The ARM instruction uses a post-indexed addressing mode, that uses a base register value as the address for the memory access, and calculates a new address from a base register value and an offset and writes it back to the base register. The offset can be an immediate value or an optionally-shifted register value.",
    [AOP7_A8_LDRD_IMMEDIATE] = "Load Register Dual (immediate) calculates an address from a base register value and an immediate offset, loads two words from memory, and writes them to two registers. It can use offset, post-indexed, or pre-indexed addressing. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDRD_LITERAL] = "Load Register Dual (literal) calculates an address from the PC value and an immediate offset, loads two words from memory, and writes them to two registers. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDRD_REGISTER] = "Load Register Dual (register) calculates an address from a base register value and a register offset, loads two words from memory, and writes them to two registers. It can use offset, post-indexed, or pre-indexed addressing. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDREX] = "Load Register Exclusive calculates an address from a base register value and an immediate offset, loads a word from memory, writes it to a register and: • if the address has the Shared Memory attribute, marks the physical address as exclusive access for the executing processor in a global monitor • causes the executing processor to indicate an active exclusive access in the local monitor. For more information about support for shared memory see Synchronization and semaphores on page A3-114. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDREXB] = "Load Register Exclusive Byte derives an address from a base register value, loads a byte from memory, zero-extends it to form a 32-bit word, writes it to a register and: • if the address has the Shared Memory attribute, marks the physical address as exclusive access for the executing processor in a global monitor • causes the executing processor to indicate an active exclusive access in the local monitor. For more information about support for shared memory see Synchronization and semaphores on page A3-114. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDREXD] = "Load Register Exclusive Doubleword derives an address from a base register value, loads a 64-bit doubleword from memory, writes it to two registers and: • if the address has the Shared Memory attribute, marks the physical address as exclusive access for the executing processor in a global monitor • causes the executing processor to indicate an active exclusive access in the local monitor. For more information about support for shared memory see Synchronization and semaphores on page A3-114. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDREXH] = "Load Register Exclusive Halfword derives an address from a base register value, loads a halfword from memory, zero-extends it to form a 32-bit word, writes it to a register and: • if the address has the Shared Memory attribute, marks the physical address as exclusive access for the executing processor in a global monitor • causes the executing processor to indicate an active exclusive access in the local monitor. For more information about support for shared memory see Synchronization and semaphores on page A3-114. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDRH_IMMEDIATE_THUMB] = "Load Register Halfword (immediate) calculates an address from a base register value and an immediate offset, loads a halfword from memory, zero-extends it to form a 32-bit word, and writes it to a register. It can use offset, post-indexed, or pre-indexed addressing. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDRH_IMMEDIATE_ARM] = "Load Register Halfword (immediate) calculates an address from a base register value and an immediate offset, loads a halfword from memory, zero-extends it to form a 32-bit word, and writes it to a register. It can use offset, post-indexed, or pre-indexed addressing. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDRH_LITERAL] = "Load Register Halfword (literal) calculates an address from the PC value and an immediate offset, loads a halfword from memory, zero-extends it to form a 32-bit word, and writes it to a register. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDRH_REGISTER] = "Load Register Halfword (register) calculates an address from a base register value and an offset register value, loads a halfword from memory, zero-extends it to form a 32-bit word, and writes it to a register. The offset register value can be shifted left by 0, 1, 2, or 3 bits. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDRHT] = "Load Register Halfword Unprivileged loads a halfword from memory, zero-extends it to form a 32-bit word, and writes it to a register. For information about memory accesses see Memory accesses on page A8-294. The memory access is restricted as if the processor were running in User mode. This makes no difference if the processor is actually running in User mode. LDRHT is UNPREDICTABLE in Hyp mode. The Thumb instruction uses an offset addressing mode, that calculates the address used for the memory access from a base register value and an immediate offset, and leaves the base register unchanged. The ARM instruction uses a post-indexed addressing mode, that uses a base register value as the address for the memory access, and calculates a new address from a base register value and an offset and writes it back to the base register. The offset can be an immediate value or a register value.",
    [AOP7_A8_LDRSB_IMMEDIATE] = "Load Register Signed Byte (immediate) calculates an address from a base register value and an immediate offset, loads a byte from memory, sign-extends it to form a 32-bit word, and writes it to a register. It can use offset, post-indexed, or pre-indexed addressing. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDRSB_LITERAL] = "Load Register Signed Byte (literal) calculates an address from the PC value and an immediate offset, loads a byte from memory, sign-extends it to form a 32-bit word, and writes it to a register. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDRSB_REGISTER] = "Load Register Signed Byte (register) calculates an address from a base register value and an offset register value, loads a byte from memory, sign-extends it to form a 32-bit word, and writes it to a register. The offset register value can be shifted left by 0, 1, 2, or 3 bits. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDRSBT] = "Load Register Signed Byte Unprivileged loads a byte from memory, sign-extends it to form a 32-bit word, and writes it to a register. For information about memory accesses see Memory accesses on page A8-294. The memory access is restricted as if the processor were running in User mode. This makes no difference if the processor is actually running in User mode. LDRSBT is UNPREDICTABLE in Hyp mode. The Thumb instruction uses an offset addressing mode, that calculates the address used for the memory access from a base register value and an immediate offset, and leaves the base register unchanged. The ARM instruction uses a post-indexed addressing mode, that uses a base register value as the address for the memory access, and calculates a new address from a base register value and an offset and writes it back to the base register. The offset can be an immediate value or a register value.",
    [AOP7_A8_LDRSH_IMMEDIATE] = "Load Register Signed Halfword (immediate) calculates an address from a base register value and an immediate offset, loads a halfword from memory, sign-extends it to form a 32-bit word, and writes it to a register. It can use offset, post-indexed, or pre-indexed addressing. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDRSH_LITERAL] = "Load Register Signed Halfword (literal) calculates an address from the PC value and an immediate offset, loads a halfword from memory, sign-extends it to form a 32-bit word, and writes it to a register. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDRSH_REGISTER] = "Load Register Signed Halfword (register) calculates an address from a base register value and an offset register value, loads a halfword from memory, sign-extends it to form a 32-bit word, and writes it to a register. The offset register value can be shifted left by 0, 1, 2, or 3 bits. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_LDRSHT] = "Load Register Signed Halfword Unprivileged loads a halfword from memory, sign-extends it to form a 32-bit word, and writes it to a register. For information about memory accesses see Memory accesses on page A8-294. The memory access is restricted as if the processor were running in User mode. This makes no difference if the processor is actually running in User mode. LDRSHT is UNPREDICTABLE in Hyp mode. The Thumb instruction uses an offset addressing mode, that calculates the address used for the memory access from a base register value and an immediate offset, and leaves the base register unchanged. The ARM instruction uses a post-indexed addressing mode, that uses a base register value as the address for the memory access, and calculates a new address from a base register value and an offset and writes it back to the base register. The offset can be an immediate value or a register value.",
    [AOP7_A8_LDRT] = "Load Register Unprivileged loads a word from memory, and writes it to a register. For information about memory accesses see Memory accesses on page A8-294. The memory access is restricted as if the processor were running in User mode. This makes no difference if the processor is actually running in User mode. LDRT is UNPREDICTABLE in Hyp mode. The Thumb instruction uses an offset addressing mode, that calculates the address used for the memory access from a base register value and an immediate offset, and leaves the base register unchanged. The ARM instruction uses a post-indexed addressing mode, that uses a base register value as the address for the memory access, and calculates a new address from a base register value and an offset and writes it back to the base register. The offset can be an immediate value or an optionally-shifted register value.",
    [AOP7_A8_LSL_IMMEDIATE] = "Logical Shift Left (immediate) shifts a register value left by an immediate number of bits, shifting in zeros, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_LSL_REGISTER] = "Logical Shift Left (register) shifts a register value left by a variable number of bits, shifting in zeros, and writes the result to the destination register. The variable number of bits is read from the bottom byte of a register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_LSR_IMMEDIATE] = "Logical Shift Right (immediate) shifts a register value right by an immediate number of bits, shifting in zeros, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_LSR_REGISTER] = "Logical Shift Right (register) shifts a register value right by a variable number of bits, shifting in zeros, and writes the result to the destination register. The variable number of bits is read from the bottom byte of a register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_MCR_MCR2] = "Move to Coprocessor from ARM core register passes the value of an ARM core register to a coprocessor. If no coprocessor can execute the instruction, an Undefined Instruction exception is generated. This is a generic coprocessor instruction. Some of the fields have no functionality defined by the architecture and are free for use by the coprocessor instruction set designer. These are the opc1, opc2, CRn, and CRm fields. However, coprocessors CP8-CP15 are reserved for use by ARM, and this manual defines the valid MCR and MCR2 instructions when coproc is in the range p8-p15. For more information see Coprocessor support on page A2-94. In an implementation that includes the Virtualization Extensions, MCR accesses to system control registers can be trapped to Hyp mode, meaning that an attempt to execute an MCR instruction in a Non-secure mode other than Hyp mode, that would be permitted in the absence of the Hyp trap controls, generates a Hyp Trap exception. For more information, see Traps to the hypervisor on page B1-1247. Note Because of the range of possible traps to Hyp mode, the MCR pseudocode does not show these possible traps.",
    [AOP7_A8_MCRR_MCRR2] = "Move to Coprocessor from two ARM core registers passes the values of two ARM core registers to a coprocessor. If no coprocessor can execute the instruction, an Undefined Instruction exception is generated. This is a generic coprocessor instruction. Some of the fields have no functionality defined by the architecture and are free for use by the coprocessor instruction set designer. These are the opc1 and CRm fields. However, coprocessors CP8-CP15 are reserved for use by ARM, and this manual defines the valid MCRR and MCRR2 instructions when coproc is in the range p8-p15. For more information see Coprocessor support on page A2-94. In an implementation that includes the Virtualization Extensions, MCRR accesses to system control registers can be trapped to Hyp mode, meaning that an attempt to execute an MCRR instruction in a Non-secure mode other than Hyp mode, that would be permitted in the absence of the Hyp trap controls, generates a Hyp Trap exception. For more information, see Traps to the hypervisor on page B1-1247. Note Because of the range of possible traps to Hyp mode, the MCRR pseudocode does not show these possible traps.",
    [AOP7_A8_MLA] = "Multiply Accumulate multiplies two register values, and adds a third register value. The least significant 32 bits of the result are written to the destination register. These 32 bits do not depend on whether the source register values are considered to be signed values or unsigned values. In an ARM instruction, the condition flags can optionally be updated based on the result. Use of this option adversely affects performance on many processor implementations.",
    [AOP7_A8_MLS] = "Multiply and Subtract multiplies two register values, and subtracts the product from a third register value. The least significant 32 bits of the result are written to the destination register. These 32 bits do not depend on whether the source register values are considered to be signed values or unsigned values.",
    [AOP7_A8_MOV_IMMEDIATE] = "Move (immediate) writes an immediate value to the destination register. It can optionally update the condition flags based on the value.",
    [AOP7_A8_MOV_REGISTER_THUMB] = "Move (register) copies a value from a register to the destination register. It can optionally update the condition flags based on the value.",
    [AOP7_A8_MOV_REGISTER_ARM] = "Move (register) copies a value from a register to the destination register. It can optionally update the condition flags based on the value.",
    [AOP7_A8_MOVT] = "Move Top writes an immediate value to the top halfword of the destination register. It does not affect the contents of the bottom halfword.",
    [AOP7_A8_MRC_MRC2] = "Move to ARM core register from Coprocessor causes a coprocessor to transfer a value to an ARM core register or to the condition flags. If no coprocessor can execute the instruction, an Undefined Instruction exception is generated. This is a generic coprocessor instruction. Some of the fields have no functionality defined by the architecture and are free for use by the coprocessor instruction set designer. These are the opc1, opc2, CRn, and CRm fields. However, coprocessors CP8-CP15 are reserved for use by ARM, and this manual defines the valid MRC and MRC2 instructions when coproc is in the range p8-p15. For more information see Coprocessor support on page A2-94. In an implementation that includes the Virtualization Extensions, MRC accesses to system control registers can be trapped to Hyp mode, meaning that an attempt to execute an MRC instruction in a Non-secure mode other than Hyp mode, that would be permitted in the absence of the Hyp trap controls, generates a Hyp Trap exception. For more information, see Traps to the hypervisor on page B1-1247. Note Because of the range of possible traps to Hyp mode, the MRC pseudocode does not show these possible traps.",
    [AOP7_A8_MRRC_MRRC2] = "Move to two ARM core registers from Coprocessor causes a coprocessor to transfer values to two ARM core registers. If no coprocessor can execute the instruction, an Undefined Instruction exception is generated. This is a generic coprocessor instruction. Some of the fields have no functionality defined by the architecture and are free for use by the coprocessor instruction set designer. These are the opc1 and CRm fields. However, coprocessors CP8-CP15 are reserved for use by ARM, and this manual defines the valid MRRC and MRRC2 instructions when coproc is in the range p8-p15. For more information see Coprocessor support on page A2-94. In an implementation that includes the Virtualization Extensions, MRRC accesses to system control registers can be trapped to Hyp mode, meaning that an attempt to execute an MRRC instruction in a Non-secure mode other than Hyp mode, that would be permitted in the absence of the Hyp trap controls, generates a Hyp Trap exception. For more information, see Traps to the hypervisor on page B1-1247. Note Because of the range of possible traps to Hyp mode, the MRRC pseudocode does not show these possible traps.",
    [AOP7_A8_MRS] = "Move to Register from Special register moves the value from the APSR into an ARM core register. For details of system level use of this instruction, see MRS on page B9-1988.",
    [AOP7_A8_MSR_IMMEDIATE] = "Move immediate value to Special register moves selected bits of an immediate value to the corresponding bits in the APSR. For details of system level use of this instruction, see MSR (immediate) on page B9-1994.",
    [AOP7_A8_MSR_REGISTER] = "Move to Special register from ARM core register moves selected bits of an ARM core register to the APSR. For details of system level use of this instruction, see MSR (register) on page B9-1996.",
    [AOP7_A8_MUL] = "Multiply multiplies two register values. The least significant 32 bits of the result are written to the destination register. These 32 bits do not depend on whether the source register values are considered to be signed values or unsigned values. Optionally, it can update the condition flags based on the result. In the Thumb instruction set, this option is limited to only a few forms of the instruction. Use of this option adversely affects performance on many processor implementations.",
    [AOP7_A8_MVN_IMMEDIATE] = "Bitwise NOT (immediate) writes the bitwise inverse of an immediate value to the destination register. It can optionally update the condition flags based on the value.",
    [AOP7_A8_MVN_REGISTER] = "Bitwise NOT (register) writes the bitwise inverse of a register value to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_MVN_REGISTER_SHIFTED_REGISTER] = "Bitwise NOT (register-shifted register) writes the bitwise inverse of a register-shifted register value to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_NOP] = "No Operation does nothing. This instruction can be used for instruction alignment purposes. See Pre-UAL pseudo-instruction NOP on page AppxH-2472 for details of NOP before the introduction of UAL and the ARMv6K and ARMv6T2 architecture variants. Note The timing effects of including a NOP instruction in a program are not guaranteed. It can increase execution time, leave it unchanged, or even reduce it. Therefore, NOP instructions are not suitable for timing loops.",
    [AOP7_A8_ORN_IMMEDIATE] = "Bitwise OR NOT (immediate) performs a bitwise (inclusive) OR of a register value and the complement of an immediate value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_ORN_REGISTER] = "Bitwise OR NOT (register) performs a bitwise (inclusive) OR of a register value and the complement of an optionally-shifted register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_ORR_IMMEDIATE] = "Bitwise OR (immediate) performs a bitwise (inclusive) OR of a register value and an immediate value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_ORR_REGISTER] = "Bitwise OR (register) performs a bitwise (inclusive) OR of a register value and an optionally-shifted register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_ORR_REGISTER_SHIFTED_REGISTER] = "Bitwise OR (register-shifted register) performs a bitwise (inclusive) OR of a register value and a register-shifted register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_PKH] = "Pack Halfword combines one halfword of its first operand with the other halfword of its shifted second operand.",
    [AOP7_A8_PLD_PLDW_IMMEDIATE] = "Preload Data signals the memory system that data memory accesses from a specified address are likely in the near future. The memory system can respond by taking actions that are expected to speed up the memory accesses when they do occur, such as pre-loading the cache line containing the specified address into the data cache. On an architecture variant that includes both the PLD and PLDW instructions, the PLD instruction signals that the likely memory access is a read, and the PLDW instruction signals that it is a write. The effect of a PLD or PLDW instruction is IMPLEMENTATION DEFINED. For more information, see Preloading caches on page A3-157 and Behavior of Preload Data (PLD, PLDW) and Preload Instruction (PLI) with caches on page B2-1269.",
    [AOP7_A8_PLD_LITERAL] = "Preload Data signals the memory system that data memory accesses from a specified address are likely in the near future. The memory system can respond by taking actions that are expected to speed up the memory accesses when they do occur, such as pre-loading the cache line containing the specified address into the data cache. The effect of a PLD instruction is IMPLEMENTATION DEFINED. For more information, see Preloading caches on page A3-157 and Behavior of Preload Data (PLD, PLDW) and Preload Instruction (PLI) with caches on page B2-1269.",
    [AOP7_A8_PLD_PLDW_REGISTER] = "Preload Data signals the memory system that data memory accesses from a specified address are likely in the near future. The memory system can respond by taking actions that are expected to speed up the memory accesses when they do occur, such as pre-loading the cache line containing the specified address into the data cache. On an architecture variant that includes both the PLD and PLDW instructions, the PLD instruction signals that the likely memory access is a read, and the PLDW instruction signals that it is a write. The effect of a PLD or PLDW instruction is IMPLEMENTATION DEFINED. For more information, see Preloading caches on page A3-157 and Behavior of Preload Data (PLD, PLDW) and Preload Instruction (PLI) with caches on page B2-1269.",
    [AOP7_A8_PLI_IMMEDIATE_LITERAL] = "Preload Instruction signals the memory system that instruction memory accesses from a specified address are likely in the near future. The memory system can respond by taking actions that are expected to speed up the memory accesses when they do occur, such as pre-loading the cache line containing the specified address into the instruction cache. The effect of a PLI instruction is IMPLEMENTATION DEFINED. For more information, see Preloading caches on page A3-157 and Behavior of Preload Data (PLD, PLDW) and Preload Instruction (PLI) with caches on page B2-1269.",
    [AOP7_A8_PLI_REGISTER] = "Preload Instruction signals the memory system that instruction memory accesses from a specified address are likely in the near future. The memory system can respond by taking actions that are expected to speed up the memory accesses when they do occur, such as pre-loading the cache line containing the specified address into the instruction cache. For more information, see Behavior of Preload Data (PLD, PLDW) and Preload Instruction (PLI) with caches on page B2-1269. The effect of a PLI instruction is IMPLEMENTATION DEFINED. For more information, see Preloading caches on page A3-157 and Behavior of Preload Data (PLD, PLDW) and Preload Instruction (PLI) with caches on page B2-1269.",
    [AOP7_A8_POP_THUMB] = "Pop Multiple Registers loads multiple registers from the stack, loading from consecutive memory locations starting at the address in SP, and updates SP to point just above the loaded data.",
    [AOP7_A8_POP_ARM] = "Pop Multiple Registers loads multiple registers from the stack, loading from consecutive memory locations starting at the address in SP, and updates SP to point just above the loaded data.",
    [AOP7_A8_PUSH] = "Push Multiple Registers stores multiple registers to the stack, storing to consecutive memory locations ending just below the address in SP, and updates SP to point to the start of the stored data.",
    [AOP7_A8_QADD] = "Saturating Add adds two register values, saturates the result to the 32-bit signed integer range –231 to (231 – 1), and writes the result to the destination register. If saturation occurs, it sets the Q flag in the APSR.",
    [AOP7_A8_QADD16] = "Saturating Add 16 performs two 16-bit integer additions, saturates the results to the 16-bit signed integer range –215 ≤ x ≤ 215 – 1, and writes the results to the destination register.",
    [AOP7_A8_QADD8] = "Saturating Add 8 performs four 8-bit integer additions, saturates the results to the 8-bit signed integer range –27 ≤ x ≤ 27 – 1, and writes the results to the destination register.",
    [AOP7_A8_QASX] = "Saturating Add and Subtract with Exchange exchanges the two halfwords of the second operand, performs one 16-bit integer addition and one 16-bit subtraction, saturates the results to the 16-bit signed integer range –215 ≤ x ≤ 215 – 1, and writes the results to the destination register.",
    [AOP7_A8_QDADD] = "Saturating Double and Add adds a doubled register value to another register value, and writes the result to the destination register. Both the doubling and the addition have their results saturated to the 32-bit signed integer range –231 ≤ x ≤ 231 – 1. If saturation occurs in either operation, it sets the Q flag in the APSR.",
    [AOP7_A8_QDSUB] = "Saturating Double and Subtract subtracts a doubled register value from another register value, and writes the result to the destination register. Both the doubling and the subtraction have their results saturated to the 32-bit signed integer range –231 ≤ x ≤ 231 – 1. If saturation occurs in either operation, it sets the Q flag in the APSR.",
    [AOP7_A8_QSAX] = "Saturating Subtract and Add with Exchange exchanges the two halfwords of the second operand, performs one 16-bit integer subtraction and one 16-bit addition, saturates the results to the 16-bit signed integer range –215 ≤ x ≤ 215 – 1, and writes the results to the destination register.",
    [AOP7_A8_QSUB] = "Saturating Subtract subtracts one register value from another register value, saturates the result to the 32-bit signed integer range –231 ≤ x ≤ 231 – 1, and writes the result to the destination register. If saturation occurs, it sets the Q flag in the APSR.",
    [AOP7_A8_QSUB16] = "Saturating Subtract 16 performs two 16-bit integer subtractions, saturates the results to the 16-bit signed integer range –215 ≤ x ≤ 215 – 1, and writes the results to the destination register.",
    [AOP7_A8_QSUB8] = "Saturating Subtract 8 performs four 8-bit integer subtractions, saturates the results to the 8-bit signed integer range –27 ≤ x ≤ 27 – 1, and writes the results to the destination register.",
    [AOP7_A8_RBIT] = "Reverse Bits reverses the bit order in a 32-bit register.",
    [AOP7_A8_REV] = "Byte-Reverse Word reverses the byte order in a 32-bit register.",
    [AOP7_A8_REV16] = "Byte-Reverse Packed Halfword reverses the byte order in each16-bit halfword of a 32-bit register.",
    [AOP7_A8_REVSH] = "Byte-Reverse Signed Halfword reverses the byte order in the lower 16-bit halfword of a 32-bit register, and sign-extends the result to 32 bits.",
    [AOP7_A8_ROR_IMMEDIATE] = "Rotate Right (immediate) provides the value of the contents of a register rotated by a constant value. The bits that are rotated off the right end are inserted into the vacated bit positions on the left. It can optionally update the condition flags based on the result.",
    [AOP7_A8_ROR_REGISTER] = "Rotate Right (register) provides the value of the contents of a register rotated by a variable number of bits. The bits that are rotated off the right end are inserted into the vacated bit positions on the left. The variable number of bits is read from the bottom byte of a register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_RRX] = "Rotate Right with Extend provides the value of the contents of a register shifted right by one place, with the Carry flag shifted into bit[31]. RRX can optionally update the condition flags based on the result. In that case, bit[0] is shifted into the Carry flag.",
    [AOP7_A8_RSB_IMMEDIATE] = "Reverse Subtract (immediate) subtracts a register value from an immediate value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_RSB_REGISTER] = "Reverse Subtract (register) subtracts a register value from an optionally-shifted register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_RSB_REGISTER_SHIFTED_REGISTER] = "Reverse Subtract (register-shifted register) subtracts a register value from a register-shifted register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_RSC_IMMEDIATE] = "Reverse Subtract with Carry (immediate) subtracts a register value and the value of NOT (Carry flag) from an immediate value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_RSC_REGISTER] = "Reverse Subtract with Carry (register) subtracts a register value and the value of NOT (Carry flag) from an optionally-shifted register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_RSC_REGISTER_SHIFTED_REGISTER] = "Reverse Subtract (register-shifted register) subtracts a register value and the value of NOT (Carry flag) from a register-shifted register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_SADD16] = "Signed Add 16 performs two 16-bit signed integer additions, and writes the results to the destination register. It sets the APSR.GE bits according to the results of the additions.",
    [AOP7_A8_SADD8] = "Signed Add 8 performs four 8-bit signed integer additions, and writes the results to the destination register. It sets the APSR.GE bits according to the results of the additions.",
    [AOP7_A8_SASX] = "Signed Add and Subtract with Exchange exchanges the two halfwords of the second operand, performs one 16-bit integer addition and one 16-bit subtraction, and writes the results to the destination register. It sets the APSR.GE bits according to the results.",
    [AOP7_A8_SBC_IMMEDIATE] = "Subtract with Carry (immediate) subtracts an immediate value and the value of NOT (Carry flag) from a register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_SBC_REGISTER] = "Subtract with Carry (register) subtracts an optionally-shifted register value and the value of NOT (Carry flag) from a register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_SBC_REGISTER_SHIFTED_REGISTER] = "Subtract with Carry (register-shifted register) subtracts a register-shifted register value and the value of NOT (Carry flag) from a register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_SBFX] = "Signed Bit Field Extract extracts any number of adjacent bits at any position from a register, sign-extends them to 32 bits, and writes the result to the destination register.",
    [AOP7_A8_SDIV] = "Signed Divide divides a 32-bit signed integer register value by a 32-bit signed integer register value, and writes the result to the destination register. The condition flags are not affected. See ARMv7 implementation requirements and options for the divide instructions on page A4-172 for more information about this instruction.",
    [AOP7_A8_SEL] = "Select Bytes selects each byte of its result from either its first operand or its second operand, according to the values of the GE flags.",
    [AOP7_A8_SETEND] = "Set Endianness writes a new value to ENDIANSTATE.",
    [AOP7_A8_SEV] = "Send Event is a hint instruction. It causes an event to be signaled to all processors in the multiprocessor system. For more information, see Wait For Event and Send Event on page B1-1199.",
    [AOP7_A8_SHADD16] = "Signed Halving Add 16 performs two signed 16-bit integer additions, halves the results, and writes the results to the destination register.",
    [AOP7_A8_SHADD8] = "Signed Halving Add 8 performs four signed 8-bit integer additions, halves the results, and writes the results to the destination register.",
    [AOP7_A8_SHASX] = "Signed Halving Add and Subtract with Exchange exchanges the two halfwords of the second operand, performs one signed 16-bit integer addition and one signed 16-bit subtraction, halves the results, and writes the results to the destination register.",
    [AOP7_A8_SHSAX] = "Signed Halving Subtract and Add with Exchange exchanges the two halfwords of the second operand, performs one signed 16-bit integer subtraction and one signed 16-bit addition, halves the results, and writes the results to the destination register.",
    [AOP7_A8_SHSUB16] = "Signed Halving Subtract 16 performs two signed 16-bit integer subtractions, halves the results, and writes the results to the destination register.",
    [AOP7_A8_SHSUB8] = "Signed Halving Subtract 8 performs four signed 8-bit integer subtractions, halves the results, and writes the results to the destination register.",
    [AOP7_A8_SMLABB_SMLABT_SMLATB_SMLATT] = "Signed Multiply Accumulate (halfwords) performs a signed multiply accumulate operation. The multiply acts on two signed 16-bit quantities, taken from either the bottom or the top half of their respective source registers. The other halves of these source registers are ignored. The 32-bit product is added to a 32-bit accumulate value and the result is written to the destination register. If overflow occurs during the addition of the accumulate value, the instruction sets the Q flag in the APSR. It is not possible for overflow to occur during the multiplication.",
    [AOP7_A8_SMLAD] = "Signed Multiply Accumulate Dual performs two signed 16 × 16-bit multiplications. It adds the products to a 32-bit accumulate operand. Optionally, the instruction can exchange the halfwords of the second operand before performing the arithmetic. This produces top × bottom and bottom × top multiplication. This instruction sets the Q flag if the accumulate operation overflows. Overflow cannot occur during the multiplications.",
    [AOP7_A8_SMLAL] = "Signed Multiply Accumulate Long multiplies two signed 32-bit values to produce a 64-bit value, and accumulates this with a 64-bit value. In ARM instructions, the condition flags can optionally be updated based on the result. Use of this option adversely affects performance on many processor implementations.",
    [AOP7_A8_SMLALBB_SMLALBT_SMLALTB_SMLALTT] = "Signed Multiply Accumulate Long (halfwords) multiplies two signed 16-bit values to produce a 32-bit value, and accumulates this with a 64-bit value. The multiply acts on two signed 16-bit quantities, taken from either the bottom or the top half of their respective source registers. The other halves of these source registers are ignored. The 32-bit product is sign-extended and accumulated with a 64-bit accumulate value. Overflow is possible during this instruction, but only as a result of the 64-bit addition. This overflow is not detected if it occurs. Instead, the result wraps around modulo 264.",
    [AOP7_A8_SMLALD] = "Signed Multiply Accumulate Long Dual performs two signed 16 × 16-bit multiplications. It adds the products to a 64-bit accumulate operand. Optionally, the instruction can exchange the halfwords of the second operand before performing the arithmetic. This produces top × bottom and bottom × top multiplication. Overflow is possible during this instruction, but only as a result of the 64-bit addition. This overflow is not detected if it occurs. Instead, the result wraps around modulo 264.",
    [AOP7_A8_SMLAWB_SMLAWT] = "Signed Multiply Accumulate (word by halfword) performs a signed multiply accumulate operation. The multiply acts on a signed 32-bit quantity and a signed 16-bit quantity. The signed 16-bit quantity is taken from either the bottom or the top half of its source register. The other half of the second source register is ignored. The top 32 bits of the 48-bit product are added to a 32-bit accumulate value and the result is written to the destination register. The bottom 16 bits of the 48-bit product are ignored. If overflow occurs during the addition of the accumulate value, the instruction sets the Q flag in the APSR. No overflow can occur during the multiplication.",
    [AOP7_A8_SMLSD] = "Signed Multiply Subtract Dual performs two signed 16 × 16-bit multiplications. It adds the difference of the products to a 32-bit accumulate operand. Optionally, the instruction can exchange the halfwords of the second operand before performing the arithmetic. This produces top × bottom and bottom × top multiplication. This instruction sets the Q flag if the accumulate operation overflows. Overflow cannot occur during the multiplications or subtraction.",
    [AOP7_A8_SMLSLD] = "Signed Multiply Subtract Long Dual performs two signed 16 × 16-bit multiplications. It adds the difference of the products to a 64-bit accumulate operand. Optionally, the instruction can exchange the halfwords of the second operand before performing the arithmetic. This produces top × bottom and bottom × top multiplication. Overflow is possible during this instruction, but only as a result of the 64-bit addition. This overflow is not detected if it occurs. Instead, the result wraps around modulo 264.",
    [AOP7_A8_SMMLA] = "Signed Most Significant Word Multiply Accumulate multiplies two signed 32-bit values, extracts the most significant 32 bits of the result, and adds an accumulate value. Optionally, the instruction can specify that the result is rounded instead of being truncated. In this case, the constant 0x80000000 is added to the product before the high word is extracted.",
    [AOP7_A8_SMMLS] = "Signed Most Significant Word Multiply Subtract multiplies two signed 32-bit values, subtracts the result from a 32-bit accumulate value that is shifted left by 32 bits, and extracts the most significant 32 bits of the result of that subtraction. Optionally, the instruction can specify that the result of the instruction is rounded instead of being truncated. In this case, the constant 0x80000000 is added to the result of the subtraction before the high word is extracted.",
    [AOP7_A8_SMMUL] = "Signed Most Significant Word Multiply multiplies two signed 32-bit values, extracts the most significant 32 bits of the result, and writes those bits to the destination register. Optionally, the instruction can specify that the result is rounded instead of being truncated. In this case, the constant 0x80000000 is added to the product before the high word is extracted.",
    [AOP7_A8_SMUAD] = "Signed Dual Multiply Add performs two signed 16 × 16-bit multiplications. It adds the products together, and writes the result to the destination register. Optionally, the instruction can exchange the halfwords of the second operand before performing the arithmetic. This produces top × bottom and bottom × top multiplication. This instruction sets the Q flag if the addition overflows. The multiplications cannot overflow.",
    [AOP7_A8_SMULBB_SMULBT_SMULTB_SMULTT] = "Signed Multiply (halfwords) multiplies two signed 16-bit quantities, taken from either the bottom or the top half of their respective source registers. The other halves of these source registers are ignored. The 32-bit product is written to the destination register. No overflow is possible during this instruction.",
    [AOP7_A8_SMULL] = "Signed Multiply Long multiplies two 32-bit signed values to produce a 64-bit result. In ARM instructions, the condition flags can optionally be updated based on the result. Use of this option adversely affects performance on many processor implementations.",
    [AOP7_A8_SMULWB_SMULWT] = "Signed Multiply (word by halfword) multiplies a signed 32-bit quantity and a signed 16-bit quantity. The signed 16-bit quantity is taken from either the bottom or the top half of its source register. The other half of the second source register is ignored. The top 32 bits of the 48-bit product are written to the destination register. The bottom 16 bits of the 48-bit product are ignored. No overflow is possible during this instruction.",
    [AOP7_A8_SMUSD] = "Signed Multiply Subtract Dual performs two signed 16 × 16-bit multiplications. It subtracts one of the products from the other, and writes the result to the destination register. Optionally, the instruction can exchange the halfwords of the second operand before performing the arithmetic. This produces top × bottom and bottom × top multiplication. Overflow cannot occur.",
    [AOP7_A8_SSAT] = "Signed Saturate saturates an optionally-shifted signed value to a selectable signed range. The Q flag is set if the operation saturates.",
    [AOP7_A8_SSAT16] = "Signed Saturate 16 saturates two signed 16-bit values to a selected signed range. The Q flag is set if the operation saturates.",
    [AOP7_A8_SSAX] = "Signed Subtract and Add with Exchange exchanges the two halfwords of the second operand, performs one 16-bit integer subtraction and one 16-bit addition, and writes the results to the destination register. It sets the APSR.GE bits according to the results.",
    [AOP7_A8_SSUB16] = "Signed Subtract 16 performs two 16-bit signed integer subtractions, and writes the results to the destination register. It sets the APSR.GE bits according to the results of the subtractions.",
    [AOP7_A8_SSUB8] = "Signed Subtract 8 performs four 8-bit signed integer subtractions, and writes the results to the destination register. It sets the APSR.GE bits according to the results of the subtractions.",
    [AOP7_A8_STC_STC2] = "Store Coprocessor stores data from a coprocessor to a sequence of consecutive memory addresses. If no coprocessor can execute the instruction, an Undefined Instruction exception is generated. This is a generic coprocessor instruction. Some of the fields have no functionality defined by the architecture and are free for use by the coprocessor instruction set designer. These are the D bit, the CRd field, and in the Unindexed addressing mode only, the imm8 field. However, coprocessors CP8-CP15 are reserved for use by ARM, and this manual defines the valid STC and STC2 instructions when coproc is in the range p8-p15. For more information see Coprocessor support on page A2-94. In an implementation that includes the Virtualization Extensions, the permitted STC access to a system control register can be trapped to Hyp mode, meaning that an attempt to execute an STC instruction in a Non-secure mode other than Hyp mode, that would be permitted in the absence of the Hyp trap controls, generates a Hyp Trap exception. For more information, see Trapping general CP14 accesses to debug registers on page B1-1260. Note For simplicity, the STC pseudocode does not show this possible trap to Hyp mode.",
    [AOP7_A8_STM_STMIA_STMEA] = "Store Multiple Increment After (Store Multiple Empty Ascending) stores multiple registers to consecutive memory locations using an address from a base register. The consecutive memory locations start at this address, and the address just above the last of those locations can optionally be written back to the base register. For details of related system instructions see STM (User registers) on page B9-2006.",
    [AOP7_A8_STMDA_STMED] = "Store Multiple Decrement After (Store Multiple Empty Descending) stores multiple registers to consecutive memory locations using an address from a base register. The consecutive memory locations end at this address, and the address just below the lowest of those locations can optionally be written back to the base register. For details of related system instructions see STM (User registers) on page B9-2006.",
    [AOP7_A8_STMDB_STMFD] = "Store Multiple Decrement Before (Store Multiple Full Descending) stores multiple registers to consecutive memory locations using an address from a base register. The consecutive memory locations end just below this address, and the address of the first of those locations can optionally be written back to the base register. For details of related system instructions see STM (User registers) on page B9-2006.",
    [AOP7_A8_STMIB_STMFA] = "Store Multiple Increment Before (Store Multiple Full Ascending) stores multiple registers to consecutive memory locations using an address from a base register. The consecutive memory locations start just above this address, and the address of the last of those locations can optionally be written back to the base register. For details of related system instructions see STM (User registers) on page B9-2006.",
    [AOP7_A8_STR_IMMEDIATE_THUMB] = "Store Register (immediate) calculates an address from a base register value and an immediate offset, and stores a word from a register to memory. It can use offset, post-indexed, or pre-indexed addressing. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_STR_IMMEDIATE_ARM] = "Store Register (immediate) calculates an address from a base register value and an immediate offset, and stores a word from a register to memory. It can use offset, post-indexed, or pre-indexed addressing. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_STR_REGISTER] = "Store Register (register) calculates an address from a base register value and an offset register value, stores a word from a register to memory. The offset register value can optionally be shifted. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_STRB_IMMEDIATE_THUMB] = "Store Register Byte (immediate) calculates an address from a base register value and an immediate offset, and stores a byte from a register to memory. It can use offset, post-indexed, or pre-indexed addressing. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_STRB_IMMEDIATE_ARM] = "Store Register Byte (immediate) calculates an address from a base register value and an immediate offset, and stores a byte from a register to memory. It can use offset, post-indexed, or pre-indexed addressing. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_STRB_REGISTER] = "Store Register Byte (register) calculates an address from a base register value and an offset register value, and stores a byte from a register to memory. The offset register value can optionally be shifted. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_STRBT] = "Store Register Byte Unprivileged stores a byte from a register to memory. For information about memory accesses see Memory accesses on page A8-294. The memory access is restricted as if the processor were running in User mode. This makes no difference if the processor is actually running in User mode. STRBT is UNPREDICTABLE in Hyp mode. The Thumb instruction uses an offset addressing mode, that calculates the address used for the memory access from a base register value and an immediate offset, and leaves the base register unchanged. The ARM instruction uses a post-indexed addressing mode, that uses a base register value as the address for the memory access, and calculates a new address from a base register value and an offset and writes it back to the base register. The offset can be an immediate value or an optionally-shifted register value.",
    [AOP7_A8_STRD_IMMEDIATE] = "Store Register Dual (immediate) calculates an address from a base register value and an immediate offset, and stores two words from two registers to memory. It can use offset, post-indexed, or pre-indexed addressing. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_STRD_REGISTER] = "Store Register Dual (register) calculates an address from a base register value and a register offset, and stores two words from two registers to memory. It can use offset, post-indexed, or pre-indexed addressing. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_STREX] = "Store Register Exclusive calculates an address from a base register value and an immediate offset, and stores a word from a register to memory if the executing processor has exclusive access to the memory addressed. For more information about support for shared memory see Synchronization and semaphores on page A3-114. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_STREXB] = "Store Register Exclusive Byte derives an address from a base register value, and stores a byte from a register to memory if the executing processor has exclusive access to the memory addressed. For more information about support for shared memory see Synchronization and semaphores on page A3-114. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_STREXD] = "Store Register Exclusive Doubleword derives an address from a base register value, and stores a 64-bit doubleword from two registers to memory if the executing processor has exclusive access to the memory addressed. For more information about support for shared memory see Synchronization and semaphores on page A3-114. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_STREXH] = "Store Register Exclusive Halfword derives an address from a base register value, and stores a halfword from a register to memory if the executing processor has exclusive access to the memory addressed. For more information about support for shared memory see Synchronization and semaphores on page A3-114. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_STRH_IMMEDIATE_THUMB] = "Store Register Halfword (immediate) calculates an address from a base register value and an immediate offset, and stores a halfword from a register to memory. It can use offset, post-indexed, or pre-indexed addressing. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_STRH_IMMEDIATE_ARM] = "Store Register Halfword (immediate) calculates an address from a base register value and an immediate offset, and stores a halfword from a register to memory. It can use offset, post-indexed, or pre-indexed addressing. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_STRH_REGISTER] = "Store Register Halfword (register) calculates an address from a base register value and an offset register value, and stores a halfword from a register to memory. The offset register value can be shifted left by 0, 1, 2, or 3 bits. For information about memory accesses see Memory accesses on page A8-294.",
    [AOP7_A8_STRHT] = "Store Register Halfword Unprivileged stores a halfword from a register to memory. For information about memory accesses see Memory accesses on page A8-294. The memory access is restricted as if the processor were running in User mode. This makes no difference if the processor is actually running in User mode. STRHT is UNPREDICTABLE in Hyp mode. The Thumb instruction uses an offset addressing mode, that calculates the address used for the memory access from a base register value and an immediate offset, and leaves the base register unchanged. The ARM instruction uses a post-indexed addressing mode, that uses a base register value as the address for the memory access, and calculates a new address from a base register value and an offset and writes it back to the base register. The offset can be an immediate value or a register value.",
    [AOP7_A8_STRT] = "Store Register Unprivileged stores a word from a register to memory. For information about memory accesses see Memory accesses on page A8-294. The memory access is restricted as if the processor were running in User mode. This makes no difference if the processor is actually running in User mode. STRT is UNPREDICTABLE in Hyp mode. The Thumb instruction uses an offset addressing mode, that calculates the address used for the memory access from a base register value and an immediate offset, and leaves the base register unchanged. The ARM instruction uses a post-indexed addressing mode, that uses a base register value as the address for the memory access, and calculates a new address from a base register value and an offset and writes it back to the base register. The offset can be an immediate value or an optionally-shifted register value.",
    [AOP7_A8_SUB_IMMEDIATE_THUMB] = "This instruction subtracts an immediate value from a register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_SUB_IMMEDIATE_ARM] = "This instruction subtracts an immediate value from a register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_SUB_REGISTER] = "This instruction subtracts an optionally-shifted register value from a register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_SUB_REGISTER_SHIFTED_REGISTER] = "This instruction subtracts a register-shifted register value from a register value, and writes the result to the destination register. It can optionally update the condition flags based on the result.",
    [AOP7_A8_SUB_SP_MINUS_IMMEDIATE] = "This instruction subtracts an immediate value from the SP value, and writes the result to the destination register.",
    [AOP7_A8_SUB_SP_MINUS_REGISTER] = "This instruction subtracts an optionally-shifted register value from the SP value, and writes the result to the destination register.",
    [AOP7_A8_SVC_PREVIOUSLY_SWI] = "Supervisor Call, previously called Software Interrupt, causes a Supervisor Call exception. For more information, see Supervisor Call (SVC) exception on page B1-1209. Software can use this instruction as a call to an operating system to provide a service. In the following cases, the Supervisor Call exception generated by the SVC instruction is taken to Hyp mode: • If the SVC is executed in Hyp mode. • If HCR.TGE is set to 1, and the SVC is executed in Non-secure User mode. For more information, see Supervisor Call exception, when HCR.TGE is set to 1 on page B1-1191 In these cases, the HSR identifies that the exception entry was caused by a Supervisor Call exception, EC value 0x11, see Use of the HSR on page B3-1424. The immediate field in the HSR: • if the SVC is unconditional: — for the Thumb instruction, is the zero-extended value of the imm8 field — for the ARM instruction, is the least-significant 16 bits the imm24 field • if the SVC is conditional, is UNKNOWN.",
    [AOP7_A8_SWP_SWPB] = "SWP (Swap) swaps a word between registers and memory. SWP loads a word from the memory address given by the value of register <Rn>. The value of register <Rt2> is then stored to the memory address given by the value of <Rn>, and the original loaded value is written to register <Rt>. If the same register is specified for <Rt> and <Rt2>, this instruction swaps the value of the register and the value at the memory address. SWPB (Swap Byte) swaps a byte between registers and memory. SWPB loads a byte from the memory address given by the value of register <Rn>. The value of the least significant byte of register <Rt2> is stored to the memory address given by <Rn>, the original loaded value is zero-extended to a 32-bit word, and the word is written to register <Rt>. If the same register is specified for <Rt> and <Rt2>, this instruction swaps the value of the least significant byte of the register and the byte value at the memory address, and clears the most significant three bytes of the register. For both instructions, the memory system ensures that no other memory access can occur to the memory location between the load access and the store access. Note • The SWP and SWPB instructions rely on the properties of the system beyond the processor to ensure that no stores from other observers can occur between the load access and the store access, and this might not be implemented for all regions of memory on some system implementations. In all cases, SWP and SWPB do ensure that no stores from the processor that executed the SWP or SWPB instruction can occur between the load access and the store access of the SWP or SWPB. • ARM deprecates the use of SWP and SWPB, and strongly recommends that new software uses: LDREX/STREX in preference to SWP — — LDREXB/STREXB in preference to SWPB. • If the translation table entries that relate to a memory location accessed by the SWP or SWPB instruction change, or are seen to change by the executing processor as a result of TLB eviction, this might mean that the translation table attributes, permissions or addresses for the load are different to those for the store. In this case, the architecture makes no guarantee that no memory access occur to these memory locations between the load and store. The Virtualization Extensions make the SWP and SWPB instructions OPTIONAL and deprecated: • If an implementation does not include the SWP and SWPB instructions, the ID_ISAR0.Swap_instrs and ID_ISAR4.SWP_frac fields are zero, see About the Instruction Set Attribute registers on page B7-1950. • In an implementation that includes SWP and SWPB, both instructions are UNDEFINED in Hyp mode.",
    [AOP7_A8_SXTAB] = "Signed Extend and Add Byte extracts an 8-bit value from a register, sign-extends it to 32 bits, adds the result to the value in another register, and writes the final result to the destination register. The instruction can specify a rotation by 0, 8, 16, or 24 bits before extracting the 8-bit value.",
    [AOP7_A8_SXTAB16] = "Signed Extend and Add Byte 16 extracts two 8-bit values from a register, sign-extends them to 16 bits each, adds the results to two 16-bit values from another register, and writes the final results to the destination register. The instruction can specify a rotation by 0, 8, 16, or 24 bits before extracting the 8-bit values.",
    [AOP7_A8_SXTAH] = "Signed Extend and Add Halfword extracts a 16-bit value from a register, sign-extends it to 32 bits, adds the result to a value from another register, and writes the final result to the destination register. The instruction can specify a rotation by 0, 8, 16, or 24 bits before extracting the 16-bit value.",
    [AOP7_A8_SXTB] = "Signed Extend Byte extracts an 8-bit value from a register, sign-extends it to 32 bits, and writes the result to the destination register. The instruction can specify a rotation by 0, 8, 16, or 24 bits before extracting the 8-bit value.",
    [AOP7_A8_SXTB16] = "Signed Extend Byte 16 extracts two 8-bit values from a register, sign-extends them to 16 bits each, and writes the results to the destination register. The instruction can specify a rotation by 0, 8, 16, or 24 bits before extracting the 8-bit values.",
    [AOP7_A8_SXTH] = "Signed Extend Halfword extracts a 16-bit value from a register, sign-extends it to 32 bits, and writes the result to the destination register. The instruction can specify a rotation by 0, 8, 16, or 24 bits before extracting the 16-bit value.",
    [AOP7_A8_TBB_TBH] = "Table Branch Byte causes a PC-relative forward branch using a table of single byte offsets. A base register provides a pointer to the table, and a second register supplies an index into the table. The branch length is twice the value of the byte returned from the table. Table Branch Halfword causes a PC-relative forward branch using a table of single halfword offsets. A base register provides a pointer to the table, and a second register supplies an index into the table. The branch length is twice the value of the halfword returned from the table.",
    [AOP7_A8_TEQ_IMMEDIATE] = "Test Equivalence (immediate) performs a bitwise exclusive OR operation on a register value and an immediate value. It updates the condition flags based on the result, and discards the result.",
    [AOP7_A8_TEQ_REGISTER] = "Test Equivalence (register) performs a bitwise exclusive OR operation on a register value and an optionally-shifted register value. It updates the condition flags based on the result, and discards the result.",
    [AOP7_A8_TEQ_REGISTER_SHIFTED_REGISTER] = "Test Equivalence (register-shifted register) performs a bitwise exclusive OR operation on a register value and a register-shifted register value. It updates the condition flags based on the result, and discards the result.",
    [AOP7_A8_TST_IMMEDIATE] = "Test (immediate) performs a bitwise AND operation on a register value and an immediate value. It updates the condition flags based on the result, and discards the result.",
    [AOP7_A8_TST_REGISTER] = "Test (register) performs a bitwise AND operation on a register value and an optionally-shifted register value. It updates the condition flags based on the result, and discards the result.",
    [AOP7_A8_TST_REGISTER_SHIFTED_REGISTER] = "Test (register-shifted register) performs a bitwise AND operation on a register value and a register-shifted register value. It updates the condition flags based on the result, and discards the result.",
    [AOP7_A8_UADD16] = "Unsigned Add 16 performs two 16-bit unsigned integer additions, and writes the results to the destination register. It sets the APSR.GE bits according to the results of the additions.",
    [AOP7_A8_UADD8] = "Unsigned Add 8 performs four unsigned 8-bit integer additions, and writes the results to the destination register. It sets the APSR.GE bits according to the results of the additions.",
    [AOP7_A8_UASX] = "Unsigned Add and Subtract with Exchange exchanges the two halfwords of the second operand, performs one unsigned 16-bit integer addition and one unsigned 16-bit subtraction, and writes the results to the destination register. It sets the APSR.GE bits according to the results.",
    [AOP7_A8_UBFX] = "Unsigned Bit Field Extract extracts any number of adjacent bits at any position from a register, zero-extends them to 32 bits, and writes the result to the destination register.",
    [AOP7_A8_UDF] = "Permanently Undefined generates an Undefined Instruction exception. The encodings for UDF used in this section are defined as permanently UNDEFINED in the versions of the architecture specified in this section. Issue C.a of this manual first defines an assembler mnemonic for these encodings. However: • with the Thumb instruction set, ARM deprecates using the UDF instruction in an IT block • in the ARM instruction set, UDF is not conditional.",
    [AOP7_A8_UDIV] = "Unsigned Divide divides a 32-bit unsigned integer register value by a 32-bit unsigned integer register value, and writes the result to the destination register. The condition flags are not affected. See ARMv7 implementation requirements and options for the divide instructions on page A4-172 for more information about this instruction.",
    [AOP7_A8_UHADD16] = "Unsigned Halving Add 16 performs two unsigned 16-bit integer additions, halves the results, and writes the results to the destination register.",
    [AOP7_A8_UHADD8] = "Unsigned Halving Add 8 performs four unsigned 8-bit integer additions, halves the results, and writes the results to the destination register.",
    [AOP7_A8_UHASX] = "Unsigned Halving Add and Subtract with Exchange exchanges the two halfwords of the second operand, performs one unsigned 16-bit integer addition and one unsigned 16-bit subtraction, halves the results, and writes the results to the destination register.",
    [AOP7_A8_UHSAX] = "Unsigned Halving Subtract and Add with Exchange exchanges the two halfwords of the second operand, performs one unsigned 16-bit integer subtraction and one unsigned 16-bit addition, halves the results, and writes the results to the destination register.",
    [AOP7_A8_UHSUB16] = "Unsigned Halving Subtract 16 performs two unsigned 16-bit integer subtractions, halves the results, and writes the results to the destination register.",
    [AOP7_A8_UHSUB8] = "Unsigned Halving Subtract 8 performs four unsigned 8-bit integer subtractions, halves the results, and writes the results to the destination register.",
    [AOP7_A8_UMAAL] = "Unsigned Multiply Accumulate Accumulate Long multiplies two unsigned 32-bit values to produce a 64-bit value, adds two unsigned 32-bit values, and writes the 64-bit result to two registers.",
    [AOP7_A8_UMLAL] = "Unsigned Multiply Accumulate Long multiplies two unsigned 32-bit values to produce a 64-bit value, and accumulates this with a 64-bit value. In ARM instructions, the condition flags can optionally be updated based on the result. Use of this option adversely affects performance on many processor implementations.",
    [AOP7_A8_UMULL] = "Unsigned Multiply Long multiplies two 32-bit unsigned values to produce a 64-bit result. In ARM instructions, the condition flags can optionally be updated based on the result. Use of this option adversely affects performance on many processor implementations.",
    [AOP7_A8_UQADD16] = "Unsigned Saturating Add 16 performs two unsigned 16-bit integer additions, saturates the results to the 16-bit unsigned integer range 0 ≤ x ≤ 216 – 1, and writes the results to the destination register.",
    [AOP7_A8_UQADD8] = "Unsigned Saturating Add 8 performs four unsigned 8-bit integer additions, saturates the results to the 8-bit unsigned integer range 0 ≤ x ≤ 28 – 1, and writes the results to the destination register.",
    [AOP7_A8_UQASX] = "Unsigned Saturating Add and Subtract with Exchange exchanges the two halfwords of the second operand, performs one unsigned 16-bit integer addition and one unsigned 16-bit subtraction, saturates the results to the 16-bit unsigned integer range 0 ≤ x ≤ 216 – 1, and writes the results to the destination register.",
    [AOP7_A8_UQSAX] = "Unsigned Saturating Subtract and Add with Exchange exchanges the two halfwords of the second operand, performs one unsigned 16-bit integer subtraction and one unsigned 16-bit addition, saturates the results to the 16-bit unsigned integer range 0 ≤ x ≤ 216 – 1, and writes the results to the destination register.",
    [AOP7_A8_UQSUB16] = "Unsigned Saturating Subtract 16 performs two unsigned 16-bit integer subtractions, saturates the results to the 16-bit unsigned integer range 0 ≤ x ≤ 216 – 1, and writes the results to the destination register.",
    [AOP7_A8_UQSUB8] = "Unsigned Saturating Subtract 8 performs four unsigned 8-bit integer subtractions, saturates the results to the 8-bit unsigned integer range 0 ≤ x ≤ 28 – 1, and writes the results to the destination register.",
    [AOP7_A8_USAD8] = "Unsigned Sum of Absolute Differences performs four unsigned 8-bit subtractions, and adds the absolute values of the differences together.",
    [AOP7_A8_USADA8] = "Unsigned Sum of Absolute Differences and Accumulate performs four unsigned 8-bit subtractions, and adds the absolute values of the differences to a 32-bit accumulate operand.",
    [AOP7_A8_USAT] = "Unsigned Saturate saturates an optionally-shifted signed value to a selected unsigned range. The Q flag is set if the operation saturates.",
    [AOP7_A8_USAT16] = "Unsigned Saturate 16 saturates two signed 16-bit values to a selected unsigned range. The Q flag is set if the operation saturates.",
    [AOP7_A8_USAX] = "Unsigned Subtract and Add with Exchange exchanges the two halfwords of the second operand, performs one unsigned 16-bit integer subtraction and one unsigned 16-bit addition, and writes the results to the destination register. It sets the APSR.GE bits according to the results.",
    [AOP7_A8_USUB16] = "Unsigned Subtract 16 performs two 16-bit unsigned integer subtractions, and writes the results to the destination register. It sets the APSR.GE bits according to the results of the subtractions.",
    [AOP7_A8_USUB8] = "Unsigned Subtract 8 performs four 8-bit unsigned integer subtractions, and writes the results to the destination register. It sets the APSR.GE bits according to the results of the subtractions.",
    [AOP7_A8_UXTAB] = "Unsigned Extend and Add Byte extracts an 8-bit value from a register, zero-extends it to 32 bits, adds the result to the value in another register, and writes the final result to the destination register. The instruction can specify a rotation by 0, 8, 16, or 24 bits before extracting the 8-bit value.",
    [AOP7_A8_UXTAB16] = "Unsigned Extend and Add Byte 16 extracts two 8-bit values from a register, zero-extends them to 16 bits each, adds the results to two 16-bit values from another register, and writes the final results to the destination register. The instruction can specify a rotation by 0, 8, 16, or 24 bits before extracting the 8-bit values.",
    [AOP7_A8_UXTAH] = "Unsigned Extend and Add Halfword extracts a 16-bit value from a register, zero-extends it to 32 bits, adds the result to a value from another register, and writes the final result to the destination register. The instruction can specify a rotation by 0, 8, 16, or 24 bits before extracting the 16-bit value.",
    [AOP7_A8_UXTB] = "Unsigned Extend Byte extracts an 8-bit value from a register, zero-extends it to 32 bits, and writes the result to the destination register. The instruction can specify a rotation by 0, 8, 16, or 24 bits before extracting the 8-bit value.",
    [AOP7_A8_UXTB16] = "Unsigned Extend Byte 16 extracts two 8-bit values from a register, zero-extends them to 16 bits each, and writes the results to the destination register. The instruction can specify a rotation by 0, 8, 16, or 24 bits before extracting the 8-bit values.",
    [AOP7_A8_UXTH] = "Unsigned Extend Halfword extracts a 16-bit value from a register, zero-extends it to 32 bits, and writes the result to the destination register. The instruction can specify a rotation by 0, 8, 16, or 24 bits before extracting the 16-bit value.",
    [AOP7_A8_VABA_VABAL] = "Vector Absolute Difference and Accumulate {Long} subtracts the elements of one vector from the corresponding elements of another vector, and accumulates the absolute values of the results into the elements of the destination vector. Operand and result elements are either all integers of the same length, or optionally the results can be double the length of the operands. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction that is not also available as a VFP instruction, see Conditional execution on page A8-288.",
    [AOP7_A8_VABD_VABDL_INTEGER] = "Vector Absolute Difference {Long} (integer) subtracts the elements of one vector from the corresponding elements of another vector, and places the absolute values of the results in the elements of the destination vector. Operand and result elements are either all integers of the same length, or optionally the results can be double the length of the operands. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction that is not also available as a VFP instruction, see Conditional execution on page A8-288.",
    [AOP7_A8_VABD_FLOATING_POINT] = "Vector Absolute Difference (floating-point) subtracts the elements of one vector from the corresponding elements of another vector, and places the absolute values of the results in the elements of the destination vector. Operand and result elements are all single-precision floating-point numbers. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction that is not also available as a VFP instruction, see Conditional execution on page A8-288.",
    [AOP7_A8_VABS] = "Vector Absolute takes the absolute value of each element in a vector, and places the results in a second vector. The floating-point version only clears the sign bit. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 and Summary of access controls for Advanced SIMD functionality on page B1-1232 summarize these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction that is not also available as a VFP instruction, see Conditional execution on page A8-288.",
    [AOP7_A8_VACGE_VACGT_VACLE_VACLT] = "VACGE (Vector Absolute Compare Greater Than or Equal) and VACGT (Vector Absolute Compare Greater Than) take the absolute value of each element in a vector, and compare it with the absolute value of the corresponding element of a second vector. If the condition is true, the corresponding element in the destination vector is set to all ones. Otherwise, it is set to all zeros. VACLE (Vector Absolute Compare Less Than or Equal) is a pseudo-instruction, equivalent to a VACGE instruction with the operands reversed. Disassembly produces the VACGE instruction. VACLT (Vector Absolute Compare Less Than) is a pseudo-instruction, equivalent to a VACGT instruction with the operands reversed. Disassembly produces the VACGT instruction. The operands and result can be quadword or doubleword vectors. They must all be the same size. The operand vector elements must be 32-bit floating-point numbers. The result vector elements are 32-bit fields. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction that is not also available as a VFP instruction, see Conditional execution on page A8-288.",
    [AOP7_A8_VADD_INTEGER] = "Vector Add adds corresponding elements in two vectors, and places the results in the destination vector. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VADD_FLOATING_POINT] = "Vector Add adds corresponding elements in two vectors, and places the results in the destination vector. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 and Summary of access controls for Advanced SIMD functionality on page B1-1232 summarize these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VADDHN] = "Vector Add and Narrow, returning High Half adds corresponding elements in two quadword vectors, and places the most significant half of each result in a doubleword vector. The results are truncated. (For rounded results, see VRADDHN on page A8-1022). The operand elements can be 16-bit, 32-bit, or 64-bit integers. There is no distinction between signed and unsigned integers. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VADDL_VADDW] = "VADDL (Vector Add Long) adds corresponding elements in two doubleword vectors, and places the results in a quadword vector. Before adding, it sign-extends or zero-extends the elements of both operands. VADDW (Vector Add Wide) adds corresponding elements in one quadword and one doubleword vector, and places the results in a quadword vector. Before adding, it sign-extends or zero-extends the elements of the doubleword operand. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VAND_REGISTER] = "This instruction performs a bitwise AND operation between two registers, and places the result in the destination register. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VBIC_IMMEDIATE] = "Vector Bitwise Bit Clear (immediate) performs a bitwise AND between a register value and the complement of an immediate value, and returns the result into the destination vector. For the range of constants available, see One register and a modified immediate value on page A7-269. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VBIC_REGISTER] = "Vector Bitwise Bit Clear (register) performs a bitwise AND between a register value and the complement of a register value, and places the result in the destination register. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VBIF_VBIT_VBSL] = "VBIF (Vector Bitwise Insert if False), VBIT (Vector Bitwise Insert if True), and VBSL (Vector Bitwise Select) perform bitwise selection under the control of a mask, and place the results in the destination register. The registers can be either quadword or doubleword, and must all be the same size. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VCEQ_REGISTER] = "VCEQ (Vector Compare Equal) takes each element in a vector, and compares it with the corresponding element of a second vector. If they are equal, the corresponding element in the destination vector is set to all ones. Otherwise, it is set to all zeros. The operand vector elements can be any one of: • 8-bit, 16-bit, or 32-bit integers. There is no distinction between signed and unsigned integers. • 32-bit floating-point numbers. The result vector elements are fields the same size as the operand vector elements. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VCEQ_IMMEDIATE_0] = "VCEQ #0 (Vector Compare Equal to zero) takes each element in a vector, and compares it with zero. If it is equal to zero, the corresponding element in the destination vector is set to all ones. Otherwise, it is set to all zeros. The operand vector elements can be any one of: • 8-bit, 16-bit, or 32-bit integers. There is no distinction between signed and unsigned integers. • 32-bit floating-point numbers. The result vector elements are fields the same size as the operand vector elements. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VCGE_REGISTER] = "VCGE (Vector Compare Greater Than or Equal) takes each element in a vector, and compares it with the corresponding element of a second vector. If the first is greater than or equal to the second, the corresponding element in the destination vector is set to all ones. Otherwise, it is set to all zeros. The operand vector elements can be any one of: • 8-bit, 16-bit, or 32-bit signed integers • 8-bit, 16-bit, or 32-bit unsigned integers • 32-bit floating-point numbers. The result vector elements are fields the same size as the operand vector elements. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VCGE_IMMEDIATE_0] = "VCGE #0 (Vector Compare Greater Than or Equal to Zero) take each element in a vector, and compares it with zero. If it is greater than or equal to zero, the corresponding element in the destination vector is set to all ones. Otherwise, it is set to all zeros. The operand vector elements can be any one of: • 8-bit, 16-bit, or 32-bit signed integers • 32-bit floating-point numbers. The result vector elements are fields the same size as the operand vector elements. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VCGT_REGISTER] = "VCGT (Vector Compare Greater Than) takes each element in a vector, and compares it with the corresponding element of a second vector. If the first is greater than the second, the corresponding element in the destination vector is set to all ones. Otherwise, it is set to all zeros. The operand vector elements can be any one of: • 8-bit, 16-bit, or 32-bit signed integers • 8-bit, 16-bit, or 32-bit unsigned integers • 32-bit floating-point numbers. The result vector elements are fields the same size as the operand vector elements. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VCGT_IMMEDIATE_0] = "VCGT #0 (Vector Compare Greater Than Zero) take each element in a vector, and compares it with zero. If it is greater than zero, the corresponding element in the destination vector is set to all ones. Otherwise, it is set to all zeros. The operand vector elements can be any one of: • 8-bit, 16-bit, or 32-bit signed integers • 32-bit floating-point numbers. The result vector elements are fields the same size as the operand vector elements. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VCLE_IMMEDIATE_0] = "VCLE #0 (Vector Compare Less Than or Equal to Zero) take each element in a vector, and compares it with zero. If it is less than or equal to zero, the corresponding element in the destination vector is set to all ones. Otherwise, it is set to all zeros. The operand vector elements can be any one of: • 8-bit, 16-bit, or 32-bit signed integers • 32-bit floating-point numbers. The result vector elements are fields the same size as the operand vector elements. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VCLS] = "Vector Count Leading Sign Bits counts the number of consecutive bits following the topmost bit, that are the same as the topmost bit, in each element in a vector, and places the results in a second vector. The count does not include the topmost bit itself. The operand vector elements can be any one of 8-bit, 16-bit, or 32-bit signed integers. The result vector elements are the same data type as the operand vector elements. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VCLT_IMMEDIATE_0] = "VCLT #0 (Vector Compare Less Than Zero) take each element in a vector, and compares it with zero. If it is less than zero, the corresponding element in the destination vector is set to all ones. Otherwise, it is set to all zeros. The operand vector elements can be any one of: • 8-bit, 16-bit, or 32-bit signed integers • 32-bit floating-point numbers. The result vector elements are fields the same size as the operand vector elements. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VCLZ] = "Vector Count Leading Zeros counts the number of consecutive zeros, starting from the most significant bit, in each element in a vector, and places the results in a second vector. The operand vector elements can be any one of 8-bit, 16-bit, or 32-bit integers. There is no distinction between signed and unsigned integers. The result vector elements are the same data type as the operand vector elements. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VCMP_VCMPE] = "This instruction compares two floating-point registers, or one floating-point register and zero. It writes the result to the FPSCR flags. These are normally transferred to the ARM flags by a subsequent VMRS instruction. It can optionally raise an Invalid Operation exception if either operand is any type of NaN. It always raises an Invalid Operation exception if either operand is a signaling NaN. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 summarizes these controls.",
    [AOP7_A8_VCNT] = "This instruction counts the number of bits that are one in each element in a vector, and places the results in a second vector. The operand vector elements must be 8-bit fields. The result vector elements are 8-bit integers. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VCVT_BETWEEN_FLOATING_POINT_AND_INTEGER_ADVANCED_SIMD] = "This instruction converts each element in a vector from floating-point to integer, or from integer to floating-point, and places the results in a second vector. The vector elements must be 32-bit floating-point numbers, or 32-bit integers. Signed and unsigned integers are distinct. The floating-point to integer operation uses the Round towards Zero rounding mode. The integer to floating-point operation uses the Round to Nearest rounding mode. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VCVT_VCVTR_BETWEEN_FLOATING_POINT_AND_INTEGER_FLOATING_POINT] = "These instructions convert a value in a register from floating-point to a 32-bit integer, or from a 32-bit integer to floating-point, and place the result in a second register. The floating-point to integer operation normally uses the Round towards Zero rounding mode, but can optionally use the rounding mode specified by the FPSCR. The integer to floating-point operation uses the rounding mode specified by the FPSCR. VCVT (between floating-point and fixed-point, Floating-point) on page A8-874 describes conversions between floating-point and 16-bit integers. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 summarizes these controls.",
    [AOP7_A8_VCVT_BETWEEN_DOUBLE_PRECISION_AND_SINGLE_PRECISION] = "This instruction does one of the following: • converts the value in a double-precision register to single-precision and writes the result to a single-precision register • converts the value in a single-precision register to double-precision and writes the result to a double-precision register. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 summarizes these controls.",
    [AOP7_A8_VCVT_BETWEEN_HALF_PRECISION_AND_SINGLE_PRECISION_ADVANCED_SIMD] = "This instruction converts each element in a vector from single-precision to half-precision floating-point or from half-precision to single-precision, and places the results in a second vector. The vector elements must be 32-bit floating-point numbers, or 16-bit floating-point numbers. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VCVTB_VCVTT] = "Vector Convert Bottom and Vector Convert Top do one of the following: • convert the half-precision value in the top or bottom half of a single-precision register to single-precision and write the result to a single-precision register • convert the value in a single-precision register to half-precision and write the result into the top or bottom half of a single-precision register, preserving the other half of the target register. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 summarizes these controls.",
    [AOP7_A8_VDIV] = "This instruction divides one floating-point value by another floating-point value and writes the result to a third floating-point register. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 summarizes these controls.",
    [AOP7_A8_VDUP_ARM_CORE_REGISTER] = "This instruction duplicates an element from an ARM core register into every element of the destination vector. The destination vector elements can be 8-bit, 16-bit, or 32-bit fields. The source element is the least significant 8, 16, or 32 bits of the ARM core register. There is no distinction between data types. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VEOR] = "Vector Bitwise Exclusive OR performs a bitwise Exclusive OR operation between two registers, and places the result in the destination register. The operand and result registers can be quadword or doubleword. They must all be the same size. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VEXT] = "Vector Extract extracts elements from the bottom end of the second operand vector and the top end of the first, concatenates them and places the result in the destination vector. See Figure A8-1 for an example. The elements of the vectors are treated as being 8-bit fields. There is no distinction between data types. 7 6 5 4 3 2 1 0 7 6 5 4 3 2 1 0 Vm Vn Vd Figure A8-1 VEXT doubleword operation for imm = 3 Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VFMA_VFMS] = "Vector Fused Multiply Accumulate multiplies corresponding elements of two vectors, and accumulates the results into the elements of the destination vector. The instruction does not round the result of the multiply before the accumulation. Vector Fused Multiply Subtract negates the elements of one vector and multiplies them with the corresponding elements of another vector, adds the products to the corresponding elements of the destination vector, and places the results in the destination vector. The instruction does not round the result of the multiply before the addition. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 and Summary of access controls for Advanced SIMD functionality on page B1-1232 summarize these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VFNMA_VFNMS] = "Vector Fused Negate Multiply Accumulate negates one floating-point register value and multiplies it by another floating-point register value, adds the negation of the floating-point value in the destination register to the product, and writes the result back to the destination register. The instruction does not round the result of the multiply before the addition. Vector Fused Negate Multiply Subtract multiplies together two floating-point register values, adds the negation of the floating-point value in the destination register to the product, and writes the result back to the destination register. The instruction does not round the result of the multiply before the addition. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 summarizes these controls.",
    [AOP7_A8_VHADD_VHSUB] = "Vector Halving Add adds corresponding elements in two vectors of integers, shifts each result right one bit, and places the final results in the destination vector. The results of the halving operations are truncated (for rounded results see VRHADD on page A8-1030). Vector Halving Subtract subtracts the elements of the second operand from the corresponding elements of the first operand, shifts each result right one bit, and places the final results in the destination vector. The results of the halving operations are truncated (there is no rounding version). The operand and result elements are all the same type, and can be any one of: • 8-bit, 16-bit, or 32-bit signed integers • 8-bit, 16-bit, or 32-bit unsigned integers. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VLD1_MULTIPLE_SINGLE_ELEMENTS] = "This instruction loads elements from memory into one, two, three, or four registers, without de-interleaving. Every element of each register is loaded. For details of the addressing mode see Advanced SIMD addressing mode on page A7-277. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VLD1_SINGLE_ELEMENT_TO_ALL_LANES] = "This instruction loads one element from memory into every element of one or two vectors. For details of the addressing mode see Advanced SIMD addressing mode on page A7-277. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VLD2_MULTIPLE_2_ELEMENT_STRUCTURES] = "This instruction loads multiple 2-element structures from memory into two or four registers, with de-interleaving. For more information, see Element and structure load/store instructions on page A4-181. Every element of each register is loaded. For details of the addressing mode see Advanced SIMD addressing mode on page A7-277. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VLD2_SINGLE_2_ELEMENT_STRUCTURE_TO_ALL_LANES] = "This instruction loads one 2-element structure from memory into all lanes of two registers. For details of the addressing mode see Advanced SIMD addressing mode on page A7-277. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VLD3_MULTIPLE_3_ELEMENT_STRUCTURES] = "This instruction loads multiple 3-element structures from memory into three registers, with de-interleaving. For more information, see Element and structure load/store instructions on page A4-181. Every element of each register is loaded. For details of the addressing mode see Advanced SIMD addressing mode on page A7-277. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VLD3_SINGLE_3_ELEMENT_STRUCTURE_TO_ALL_LANES] = "This instruction loads one 3-element structure from memory into all lanes of three registers. For details of the addressing mode see Advanced SIMD addressing mode on page A7-277. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VLD4_MULTIPLE_4_ELEMENT_STRUCTURES] = "This instruction loads multiple 4-element structures from memory into four registers, with de-interleaving. For more information, see Element and structure load/store instructions on page A4-181. Every element of each register is loaded. For details of the addressing mode see Advanced SIMD addressing mode on page A7-277. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VLD4_SINGLE_4_ELEMENT_STRUCTURE_TO_ALL_LANES] = "This instruction loads one 4-element structure from memory into all lanes of four registers. For details of the addressing mode see Advanced SIMD addressing mode on page A7-277. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VLDM] = "Vector Load Multiple loads multiple extension registers from consecutive memory locations using an address from an ARM core register. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 summarizes these controls.",
    [AOP7_A8_VMAX_VMIN_INTEGER] = "Vector Maximum compares corresponding elements in two vectors, and copies the larger of each pair into the corresponding element in the destination vector. Vector Minimum compares corresponding elements in two vectors, and copies the smaller of each pair into the corresponding element in the destination vector. The operand vector elements can be any one of: • 8-bit, 16-bit, or 32-bit signed integers • 8-bit, 16-bit, or 32-bit unsigned integers. The result vector elements are the same size as the operand vector elements. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VMAX_VMIN_FLOATING_POINT] = "Vector Maximum compares corresponding elements in two vectors, and copies the larger of each pair into the corresponding element in the destination vector. Vector Minimum compares corresponding elements in two vectors, and copies the smaller of each pair into the corresponding element in the destination vector. The operand vector elements are 32-bit floating-point numbers. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VMLA_VMLAL_VMLS_VMLSL_INTEGER] = "Vector Multiply Accumulate and Vector Multiply Subtract multiply corresponding elements in two vectors, and either add the products to, or subtract them from, the corresponding elements of the destination vector. Vector Multiply Accumulate Long and Vector Multiply Subtract Long do the same thing, but with destination vector elements that are twice as long as the elements that are multiplied. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VMLA_VMLS_FLOATING_POINT] = "Vector Multiply Accumulate multiplies corresponding elements in two vectors, and accumulates the results into the elements of the destination vector. Vector Multiply Subtract multiplies corresponding elements in two vectors, subtracts the products from corresponding elements of the destination vector, and places the results in the destination vector. Note ARM recommends that software does not use the VMLS instruction in the Round towards Plus Infinity and Round towards Minus Infinity rounding modes, because the rounding of the product and of the sum can change the result of the instruction in opposite directions, defeating the purpose of these rounding modes. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 and Summary of access controls for Advanced SIMD functionality on page B1-1232 summarize these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VMOV_BETWEEN_ARM_CORE_REGISTER_AND_SINGLE_PRECISION_REGISTER] = "This instruction transfers the contents of a single-precision Floating-point register to an ARM core register, or the contents of an ARM core register to a single-precision Floating-point register. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 summarizes these controls.",
    [AOP7_A8_VMOV_BETWEEN_TWO_ARM_CORE_REGISTERS_AND_TWO_SINGLE_PRECISION_REGISTERS] = "This instruction transfers the contents of two consecutively numbered single-precision Floating-point registers to two ARM core registers, or the contents of two ARM core registers to a pair of single-precision Floating-point registers. The ARM core registers do not have to be contiguous. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 summarizes these controls.",
    [AOP7_A8_VMOV_BETWEEN_TWO_ARM_CORE_REGISTERS_AND_A_DOUBLEWORD_EXTENSION_REGISTER] = "This instruction copies two words from two ARM core registers into a doubleword extension register, or from a doubleword extension register to two ARM core registers. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 and Summary of access controls for Advanced SIMD functionality on page B1-1232 summarize these controls.",
    [AOP7_A8_VMOVL] = "Vector Move Long takes each element in a doubleword vector, sign or zero-extends them to twice their original length, and places the results in a quadword vector. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VMOVN] = "Vector Move and Narrow copies the least significant half of each element of a quadword vector into the corresponding elements of a doubleword vector. The operand vector elements can be any one of 16-bit, 32-bit, or 64-bit integers. There is no distinction between signed and unsigned integers. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VMRS] = "Move to ARM core register from Advanced SIMD and Floating-point Extension System Register moves the value of the FPSCR to an ARM core register. For details of system level use of this instruction, see VMRS on page B9-2012. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 and Summary of access controls for Advanced SIMD functionality on page B1-1232 summarize these controls.",
    [AOP7_A8_VMSR] = "Move to Advanced SIMD and Floating-point Extension System Register from ARM core register moves the value of an ARM core register to the FPSCR. For details of system level use of this instruction, see VMSR on page B9-2014. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 and Summary of access controls for Advanced SIMD functionality on page B1-1232 summarize these controls.",
    [AOP7_A8_VMUL_VMULL_INTEGER_AND_POLYNOMIAL] = "Vector Multiply multiplies corresponding elements in two vectors. Vector Multiply Long does the same thing, but with destination vector elements that are twice as long as the elements that are multiplied. For information about multiplying polynomials see Polynomial arithmetic over {0, 1} on page A2-93. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VMUL_FLOATING_POINT] = "Vector Multiply multiplies corresponding elements in two vectors, and places the results in the destination vector. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 and Summary of access controls for Advanced SIMD functionality on page B1-1232 summarize these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VMVN_IMMEDIATE] = "Vector Bitwise NOT (immediate) places the bitwise inverse of an immediate integer constant into every element of the destination register. For the range of constants available, see One register and a modified immediate value on page A7-269. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VMVN_REGISTER] = "Vector Bitwise NOT (register) takes a value from a register, inverts the value of each bit, and places the result in the destination register. The registers can be either doubleword or quadword. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VNEG] = "Vector Negate negates each element in a vector, and places the results in a second vector. The floating-point version only inverts the sign bit. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 and Summary of access controls for Advanced SIMD functionality on page B1-1232 summarize these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VNMLA_VNMLS_VNMUL] = "VNMLA multiplies together two floating-point register values, adds the negation of the floating-point value in the destination register to the negation of the product, and writes the result back to the destination register. VNMLS multiplies together two floating-point register values, adds the negation of the floating-point value in the destination register to the product, and writes the result back to the destination register. VNMUL multiplies together two floating-point register values, and writes the negation of the result to the destination register. Note ARM recommends that software does not use the VNMLA instruction in the Round towards Plus Infinity and Round towards Minus Infinity rounding modes, because the rounding of the product and of the sum can change the result of the instruction in opposite directions, defeating the purpose of these rounding modes. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 summarizes these controls.",
    [AOP7_A8_VORN_REGISTER] = "This instruction performs a bitwise OR NOT operation between two registers, and places the result in the destination register. The operand and result registers can be quadword or doubleword. They must all be the same size. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VORR_IMMEDIATE] = "This instruction takes the contents of the destination vector, performs a bitwise OR with an immediate constant, and returns the result into the destination vector. For the range of constants available, see One register and a modified immediate value on page A7-269. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VORR_REGISTER] = "This instruction performs a bitwise OR operation between two registers, and places the result in the destination register. The operand and result registers can be quadword or doubleword. They must all be the same size. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VPADAL] = "Vector Pairwise Add and Accumulate Long adds adjacent pairs of elements of a vector, and accumulates the results into the elements of the destination vector. The vectors can be doubleword or quadword. The operand elements can be 8-bit, 16-bit, or 32-bit integers. The result elements are twice the length of the operand elements. Figure A8-2 shows an example of the operation of VPADAL. Dm + + Dd Figure A8-2 VPADAL doubleword operation for data type S16 Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VPADD_INTEGER] = "Vector Pairwise Add (integer) adds adjacent pairs of elements of two vectors, and places the results in the destination vector. The operands and result are doubleword vectors. The operand and result elements must all be the same type, and can be 8-bit, 16-bit, or 32-bit integers. There is no distinction between signed and unsigned integers. Figure A8-3 shows an example of the operation of VPADD. Dm Dn + + + + Dd Figure A8-3 VPADD operation for data type I16 Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VPADD_FLOATING_POINT] = "Vector Pairwise Add (floating-point) adds adjacent pairs of elements of two vectors, and places the results in the destination vector. The operands and result are doubleword vectors. The operand and result elements are 32-bit floating-point numbers. Figure A8-3 on page A8-980 shows an example of the operation of VPADD. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VPADDL] = "Vector Pairwise Add Long adds adjacent pairs of elements of two vectors, and places the results in the destination vector. The vectors can be doubleword or quadword. The operand elements can be 8-bit, 16-bit, or 32-bit integers. The result elements are twice the length of the operand elements. Figure A8-4 shows an example of the operation of VPADDL. Dm + + Dd Figure A8-4 VPADDL doubleword operation for data type S16 Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VPMAX_VPMIN_INTEGER] = "Vector Pairwise Maximum compares adjacent pairs of elements in two doubleword vectors, and copies the larger of each pair into the corresponding element in the destination doubleword vector. Vector Pairwise Minimum compares adjacent pairs of elements in two doubleword vectors, and copies the smaller of each pair into the corresponding element in the destination doubleword vector. Figure A8-5 shows an example of the operation of VPMAX. Dm Dn max max max max Dd Figure A8-5 VPMAX operation for data type S16 or U16 Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VPMAX_VPMIN_FLOATING_POINT] = "Vector Pairwise Maximum compares adjacent pairs of elements in two doubleword vectors, and copies the larger of each pair into the corresponding element in the destination doubleword vector. Vector Pairwise Minimum compares adjacent pairs of elements in two doubleword vectors, and copies the smaller of each pair into the corresponding element in the destination doubleword vector. Figure A8-5 on page A8-986 shows an example of the operation of VPMAX. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VPOP] = "Vector Pop loads multiple consecutive extension registers from the stack. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 and Summary of access controls for Advanced SIMD functionality on page B1-1232 summarize these controls.",
    [AOP7_A8_VPUSH] = "Vector Push stores multiple consecutive extension registers to the stack. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 and Summary of access controls for Advanced SIMD functionality on page B1-1232 summarize these controls.",
    [AOP7_A8_VQABS] = "Vector Saturating Absolute takes the absolute value of each element in a vector, and places the results in the destination vector. If any of the results overflow, they are saturated. The cumulative saturation bit, FPSCR.QC, is set if saturation occurs. For details see Pseudocode details of saturation on page A2-44. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VQADD] = "Vector Saturating Add adds the values of corresponding elements of two vectors, and places the results in the destination vector. If any of the results overflow, they are saturated. The cumulative saturation bit, FPSCR.QC, is set if saturation occurs. For details see Pseudocode details of saturation on page A2-44. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VQMOVN_VQMOVUN] = "Vector Saturating Move and Narrow copies each element of the operand vector to the corresponding element of the destination vector. The operand is a quadword vector. The elements can be any one of: • 16-bit, 32-bit, or 64-bit signed integers • 16-bit, 32-bit, or 64-bit unsigned integers. The result is a doubleword vector. The elements are half the length of the operand vector elements. If the operand is unsigned, the results are unsigned. If the operand is signed, the results can be signed or unsigned. If any of the results overflow, they are saturated. The cumulative saturation bit, FPSCR.QC, is set if saturation occurs. For details see Pseudocode details of saturation on page A2-44. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VQNEG] = "Vector Saturating Negate negates each element in a vector, and places the results in the destination vector. If any of the results overflow, they are saturated. The cumulative saturation bit, FPSCR.QC, is set if saturation occurs. For details see Pseudocode details of saturation on page A2-44. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VQRSHL] = "Vector Saturating Rounding Shift Left takes each element in a vector, shifts them by a value from the least significant byte of the corresponding element of a second vector, and places the results in the destination vector. If the shift value is positive, the operation is a left shift. Otherwise, it is a right shift. For truncated results see VQSHL (register) on page A8-1014. The first operand and result elements are the same data type, and can be any one of: • 8-bit, 16-bit, 32-bit, or 64-bit signed integers • 8-bit, 16-bit, 32-bit, or 64-bit unsigned integers. The second operand is a signed integer of the same size. If any of the results overflow, they are saturated. The cumulative saturation bit, FPSCR.QC, is set if saturation occurs. For details see Pseudocode details of saturation on page A2-44. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VQSHL_REGISTER] = "Vector Saturating Shift Left (register) takes each element in a vector, shifts them by a value from the least significant byte of the corresponding element of a second vector, and places the results in the destination vector. If the shift value is positive, the operation is a left shift. Otherwise, it is a right shift. The results are truncated. For rounded results, see VQRSHL on page A8-1010. The first operand and result elements are the same data type, and can be any one of: • 8-bit, 16-bit, 32-bit, or 64-bit signed integers • 8-bit, 16-bit, 32-bit, or 64-bit unsigned integers. The second operand is a signed integer of the same size. If any of the results overflow, they are saturated. The cumulative saturation bit, FPSCR.QC, is set if saturation occurs. For details see Pseudocode details of saturation on page A2-44. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VQSUB] = "Vector Saturating Subtract subtracts the elements of the second operand vector from the corresponding elements of the first operand vector, and places the results in the destination vector. Signed and unsigned operations are distinct. The operand and result elements must all be the same type, and can be any one of: • 8-bit, 16-bit, 32-bit, or 64-bit signed integers • 8-bit, 16-bit, 32-bit, or 64-bit unsigned integers. If any of the results overflow, they are saturated. The cumulative saturation bit, FPSCR.QC, is set if saturation occurs. For details see Pseudocode details of saturation on page A2-44. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VRADDHN] = "Vector Rounding Add and Narrow, returning High Half adds corresponding elements in two quadword vectors, and places the most significant half of each result in a doubleword vector. The results are rounded. (For truncated results, see VADDHN on page A8-832.) The operand elements can be 16-bit, 32-bit, or 64-bit integers. There is no distinction between signed and unsigned integers. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VRECPE] = "Vector Reciprocal Estimate finds an approximate reciprocal of each element in the operand vector, and places the results in the destination vector. The operand and result elements are the same type, and can be 32-bit floating-point numbers, or 32-bit unsigned integers. For details of the operation performed by this instruction see Floating-point reciprocal estimate and step on page A2-85. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VRECPS] = "Vector Reciprocal Step multiplies the elements of one vector by the corresponding elements of another vector, subtracts each of the products from 2.0, and places the results into the elements of the destination vector. The operand and result elements are 32-bit floating-point numbers. For details of the operation performed by this instruction see Floating-point reciprocal estimate and step on page A2-85. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VREV16_VREV32_VREV64] = "VREV16 (Vector Reverse in halfwords) reverses the order of 8-bit elements in each halfword of the vector, and places the result in the corresponding destination vector. VREV32 (Vector Reverse in words) reverses the order of 8-bit or 16-bit elements in each word of the vector, and places the result in the corresponding destination vector. VREV64 (Vector Reverse in doublewords) reverses the order of 8-bit, 16-bit, or 32-bit elements in each doubleword of the vector, and places the result in the corresponding destination vector. There is no distinction between data types, other than size. Figure A8-6 shows two examples of the operation of VREV. VREV64.8, doubleword VREV64.32, quadword Dm Qm Dd Qm Figure A8-6 VREV operation examples Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VRHADD] = "Vector Rounding Halving Add adds corresponding elements in two vectors of integers, shifts each result right one bit, and places the final results in the destination vector. The operand and result elements are all the same type, and can be any one of: • 8-bit, 16-bit, or 32-bit signed integers • 8-bit, 16-bit, or 32-bit unsigned integers. The results of the halving operations are rounded. For truncated results see VHADD, VHSUB on page A8-896. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VRSHL] = "Vector Rounding Shift Left takes each element in a vector, shifts them by a value from the least significant byte of the corresponding element of a second vector, and places the results in the destination vector. If the shift value is positive, the operation is a left shift. If the shift value is negative, it is a rounding right shift. (For a truncating shift, see VSHL (register) on page A8-1048). The first operand and result elements are the same data type, and can be any one of: • 8-bit, 16-bit, 32-bit, or 64-bit signed integers • 8-bit, 16-bit, 32-bit, or 64-bit unsigned integers. The second operand is always a signed integer of the same size. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VRSQRTE] = "Vector Reciprocal Square Root Estimate finds an approximate reciprocal square root of each element in a vector, and places the results in a second vector. The operand and result elements are the same type, and can be 32-bit floating-point numbers, or 32-bit unsigned integers. For details of the operation performed by this instruction see Floating-point reciprocal square root estimate and step on page A2-87. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VRSQRTS] = "Vector Reciprocal Square Root Step multiplies the elements of one vector by the corresponding elements of another vector, subtracts each of the products from 3.0, divides these results by 2.0, and places the results into the elements of the destination vector. The operand and result elements are 32-bit floating-point numbers. For details of the operation performed by this instruction see Floating-point reciprocal square root estimate and step on page A2-87. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VRSUBHN] = "Vector Rounding Subtract and Narrow, returning High Half subtracts the elements of one quadword vector from the corresponding elements of another quadword vector takes the most significant half of each result, and places the final results in a doubleword vector. The results are rounded. (For truncated results, see VSUBHN on page A8-1088.) The operand elements can be 16-bit, 32-bit, or 64-bit integers. There is no distinction between signed and unsigned integers. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VSHL_REGISTER] = "Vector Shift Left (register) takes each element in a vector, shifts them by a value from the least significant byte of the corresponding element of a second vector, and places the results in the destination vector. If the shift value is positive, the operation is a left shift. If the shift value is negative, it is a truncating right shift. Note For a rounding shift, see VRSHL on page A8-1032. The first operand and result elements are the same data type, and can be any one of: • 8-bit, 16-bit, 32-bit, or 64-bit signed integers • 8-bit, 16-bit, 32-bit, or 64-bit unsigned integers. The second operand is always a signed integer of the same size. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VSQRT] = "This instruction calculates the square root of the value in a floating-point register and writes the result to another floating-point register. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 summarizes these controls.",
    [AOP7_A8_VST1_MULTIPLE_SINGLE_ELEMENTS] = "Vector Store (multiple single elements) stores elements to memory from one, two, three, or four registers, without interleaving. Every element of each register is stored. For details of the addressing mode see Advanced SIMD addressing mode on page A7-277. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VST2_MULTIPLE_2_ELEMENT_STRUCTURES] = "This instruction stores multiple 2-element structures from two or four registers to memory, with interleaving. For more information, see Element and structure load/store instructions on page A4-181. Every element of each register is saved. For details of the addressing mode see Advanced SIMD addressing mode on page A7-277. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VST3_MULTIPLE_3_ELEMENT_STRUCTURES] = "This instruction stores multiple 3-element structures to memory from three registers, with interleaving. For more information, see Element and structure load/store instructions on page A4-181. Every element of each register is saved. For details of the addressing mode see Advanced SIMD addressing mode on page A7-277. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VST4_MULTIPLE_4_ELEMENT_STRUCTURES] = "This instruction stores multiple 4-element structures to memory from four registers, with interleaving. For more information, see Element and structure load/store instructions on page A4-181. Every element of each register is saved. For details of the addressing mode see Advanced SIMD addressing mode on page A7-277. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VSTM] = "Vector Store Multiple stores multiple extension registers to consecutive memory locations using an address from an ARM core register. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 and Summary of access controls for Advanced SIMD functionality on page B1-1232 summarize these controls.",
    [AOP7_A8_VSTR] = "This instruction stores a single extension register to memory, using an address from an ARM core register, with an optional offset. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 and Summary of access controls for Advanced SIMD functionality on page B1-1232 summarize these controls.",
    [AOP7_A8_VSUB_INTEGER] = "Vector Subtract subtracts the elements of one vector from the corresponding elements of another vector, and places the results in the destination vector. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VSUB_FLOATING_POINT] = "Vector Subtract subtracts the elements of one vector from the corresponding elements of another vector, and places the results in the destination vector. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 and Summary of access controls for Advanced SIMD functionality on page B1-1232 summarize these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VSUBHN] = "Vector Subtract and Narrow, returning High Half subtracts the elements of one quadword vector from the corresponding elements of another quadword vector, takes the most significant half of each result, and places the final results in a doubleword vector. The results are truncated. (For rounded results, see VRSUBHN on page A8-1044. There is no distinction between signed and unsigned integers. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VSUBL_VSUBW] = "Vector Subtract Long subtracts the elements of one doubleword vector from the corresponding elements of another doubleword vector, and places the results in a quadword vector. Before subtracting, it sign-extends or zero-extends the elements of both operands. Vector Subtract Wide subtracts the elements of a doubleword vector from the corresponding elements of a quadword vector, and places the results in another quadword vector. Before subtracting, it sign-extends or zero-extends the elements of the doubleword operand. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VSWP] = "VSWP (Vector Swap) exchanges the contents of two vectors. The vectors can be either doubleword or quadword. There is no distinction between data types. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VTBL_VTBX] = "Vector Table Lookup uses byte indexes in a control vector to look up byte values in a table and generate a new vector. Indexes out of range return 0. Vector Table Extension works in the same way, except that indexes out of range leave the destination element unchanged. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VTRN] = "Vector Transpose treats the elements of its operand vectors as elements of 2 × 2 matrices, and transposes the matrices. The elements of the vectors can be 8-bit, 16-bit, or 32-bit. There is no distinction between data types. Figure A8-7 shows the operation of doubleword VTRN. Quadword VTRN performs the same operation as doubleword VTRN twice, once on the upper halves of the quadword vectors, and once on the lower halves VTRN.32 VTRN.16 VTRN.8 1 0 3 2 1 0 7 6 5 4 3 2 1 0 Dd Dd Dd Dm Dm Dm Figure A8-7 VTRN doubleword operation Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VTST] = "Vector Test Bits takes each element in a vector, and bitwise ANDs it with the corresponding element of a second vector. If the result is not zero, the corresponding element in the destination vector is set to all ones. Otherwise, it is set to all zeros. The operand vector elements can be any one of: • 8-bit, 16-bit, or 32-bit fields. The result vector elements are fields the same size as the operand vector elements. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VUZP] = "Vector Unzip de-interleaves the elements of two vectors. See Table A8-13 and Table A8-14 for examples of the operation. The elements of the vectors can be 8-bit, 16-bit, or 32-bit. There is no distinction between data types. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_VZIP] = "Vector Zip interleaves the elements of two vectors. See Table A8-15 and Table A8-16 for examples of the operation. The elements of the vectors can be 8-bit, 16-bit, or 32-bit. There is no distinction between data types. Depending on settings in the CPACR, NSACR, and HCPTR registers, and the security state and mode in which the instruction is executed, an attempt to execute the instruction might be UNDEFINED, or trapped to Hyp mode. Summary of access controls for Advanced SIMD functionality on page B1-1232 summarizes these controls. ARM deprecates the conditional execution of any Advanced SIMD instruction encoding that is not also available as a VFP instruction encoding, see Conditional execution on page A8-288.",
    [AOP7_A8_WFE] = "Wait For Event is a hint instruction that permits the processor to enter a low-power state until one of a number of events occurs, including events signaled by executing the SEV instruction on any processor in the multiprocessor system. For more information, see Wait For Event and Send Event on page B1-1199. In an implementation that includes the Virtualization Extensions, if HCR.TWE is set to 1, execution of a WFE instruction in a Non-secure mode other than Hyp mode generates a Hyp Trap exception if, ignoring the value of the HCR.TWE bit, conditions permit the processor to suspend execution. For more information see Trapping use of the WFI and WFE instructions on page B1-1255.",
    [AOP7_A8_WFI] = "Wait For Interrupt is a hint instruction that permits the processor to enter a low-power state until one of a number of asynchronous events occurs. For more information, see Wait For Interrupt on page B1-1202. In an implementation that includes the Virtualization Extensions, if HCR.TWI is set to 1, execution of a WFI instruction in a Non-secure mode other than Hyp mode generates a Hyp Trap exception if, ignoring the value of the HCR.TWI bit, conditions permit the processor to suspend execution. For more information see Trapping use of the WFI and WFE instructions on page B1-1255.",
    [AOP7_A8_YIELD] = "YIELD is a hint instruction. Software with a multithreading capability can use a YIELD instruction to indicate to the hardware that it is performing a task, for example a spin-lock, that could be swapped out to improve overall system performance. Hardware can use this hint to suspend and resume multiple software threads if it supports the capability. For more information about the recommended use of this instruction see The Yield instruction on page A4-178.",
    [AOP7_A9_ENTERX_LEAVEX] = "ENTERX causes a change from Thumb state to ThumbEE state, or has no effect in ThumbEE state. ENTERX is UNDEFINED in Hyp mode. LEAVEX causes a change from ThumbEE state to Thumb state, or has no effect in Thumb state.",
    [AOP7_B9_CPS_THUMB] = "Change Processor State changes one or more of the CPSR.{A, I, F} interrupt mask bits and the CPSR.M mode field, without changing the other CPSR bits. CPS is treated as NOP if executed in User mode. CPS is UNPREDICTABLE if it is either: • attempting to change to a mode that is not permitted in the context in which it is executed, see Restrictions on updates to the CPSR.M field on page B9-1970 • executed in Debug state.",
    [AOP7_B9_CPS_ARM] = "Change Processor State changes one or more of the CPSR.{A, I, F} interrupt mask bits and the CPSR.M mode field, without changing the other CPSR bits. CPS is treated as NOP if executed in User mode. CPS is UNPREDICTABLE if it is either: • attempting to change to a mode that is not permitted in the context in which it is executed, see Restrictions on updates to the CPSR.M field on page B9-1970 • executed in Debug state.",
    [AOP7_B9_ERET] = "When executed in Hyp mode, Exception Return loads the PC from ELR_hyp and loads the CPSR from SPSR_hyp. When executed in a Secure or Non-secure PL1 mode, ERET behaves as: MOVS PC, • LR in the ARM instruction set, see SUBS PC, LR and related instructions (ARM) on page B9-2010 • the equivalent SUBS PC, LR, #0 in the Thumb instruction set, see SUBS PC, LR (Thumb) on page B9-2008. ERET is UNPREDICTABLE: • in the cases described in Restrictions on exception return instructions on page B9-1970 • if it is executed in Debug state. Note In an implementation that includes the Virtualization Extensions: • LR, #0 in the Thumb The T1 encoding of ERET is not a new encoding but, is the preferred synonym of SUBS PC, instruction set. See SUBS PC, LR (Thumb) on page B9-2008 for more information. • Because ERET is the preferred encoding, when decoding Thumb instructions, a disassembler will report an ERET where the original assembler code used SUBS PC, LR, #0.",
    [AOP7_B9_HVC] = "Hypervisor Call causes a Hypervisor Call exception. For more information see Hypervisor Call (HVC) exception on page B1-1211. Non-secure software executing at PL1 can use this instruction to call the hypervisor to request a service. The HVC instruction is: • UNDEFINED in Secure state, and in User mode in Non-secure state • when SCR.HCE is set to 0, UNDEFINED in Non-secure PL1 modes and UNPREDICTABLE in Hyp mode • UNPREDICTABLE in Debug state. On executing an HVC instruction, the HSR reports the exception as a Hypervisor Call exception, using the EC value 0x12, and captures the value of the immediate argument, see Use of the HSR on page B3-1424.",
    [AOP7_B9_LDM_EXCEPTION_RETURN] = "Load Multiple (exception return) loads multiple registers from consecutive memory locations using an address from a base register. The SPSR of the current mode is copied to the CPSR. An address adjusted by the size of the data loaded can optionally be written back to the base register. The registers loaded include the PC. The word loaded for the PC is treated as an address and a branch occurs to that address. LDM (exception return) is: • UNDEFINED in Hyp mode • UNPREDICTABLE in: — the cases described in Restrictions on exception return instructions on page B9-1970 — Debug state.",
    [AOP7_B9_LDM_USER_REGISTERS] = "In a PL1 mode other than System mode, Load Multiple (User registers) loads multiple User mode registers from consecutive memory locations using an address from a base register. The registers loaded cannot include the PC. The processor reads the base register value normally, using the current mode to determine the correct Banked version of the register. This instruction cannot writeback to the base register. LDM (user registers) is UNDEFINED in Hyp mode, and UNPREDICTABLE in User and System modes.",
    [AOP7_B9_MRS] = "Move to Register from Special register moves the value from the CPSR or SPSR of the current mode into an ARM core register. An MRS that accesses the SPSR is UNPREDICTABLE if executed in User or System mode. An MRS that is executed in User mode and accesses the CPSR returns an UNKNOWN value for the CPSR.{E, A, I, F, M} fields. Note MRS on page A8-496 describes the valid application level uses of the MRS instruction.",
    [AOP7_B9_MRS_BANKED_REGISTER] = "Move to Register from Banked or Special register moves the value from the Banked ARM core register or SPSR of the specified mode, or the value of ELR_hyp, to an ARM core register. MRS (Banked register) is UNPREDICTABLE if executed in User mode. The effect of using an MRS (Banked register) instruction with a register argument that is not valid for the current mode is UNPREDICTABLE. For more information see Usage restrictions on the Banked register transfer instructions on page B9-1972.",
    [AOP7_B9_MSR_BANKED_REGISTER] = "Move to Banked or Special register from ARM core register moves the value of an ARM core register to the Banked ARM core register or SPSR of the specified mode, or to ELR_hyp. MSR (Banked register) is UNPREDICTABLE if executed in User mode. The effect of using an MSR (Banked register) instruction with a register argument that is not valid for the current mode is UNPREDICTABLE. For more information see Usage restrictions on the Banked register transfer instructions on page B9-1972.",
    [AOP7_B9_MSR_IMMEDIATE] = "Move immediate value to Special register moves selected bits of an immediate value to the CPSR or the SPSR of the current mode. MSR (immediate) is UNPREDICTABLE if: • In Non-debug state, it is attempting to update the CPSR, and that update would change to a mode that is not permitted in the context in which the instruction is executed, see Restrictions on updates to the CPSR.M field on page B9-1970. • In Debug state, it is attempting an update to the CPSR with a <fields> value that is not <fsxc>. See Behavior of MRS and MSR instructions that access the CPSR in Debug state on page C5-2097. An MSR (immediate) executed in User mode: • is UNPREDICTABLE if it attempts to update the SPSR • otherwise, does not update any CPSR field that is accessible only at PL1 or higher, Note MSR (immediate) on page A8-498 describes the valid application level uses of the MSR (immediate) instruction. An MSR (immediate) executed in System mode is UNPREDICTABLE if it attempts to update the SPSR.",
    [AOP7_B9_MSR_REGISTER] = "Move to Special register from ARM core register moves the value of an ARM core register to the CPSR or the SPSR of the current mode. MSR (register) is UNPREDICTABLE if: • In Non-debug state, it is attempting to update the CPSR, and that update would change to a mode that is not permitted in the context in which the instruction is executed, see Restrictions on updates to the CPSR.M field on page B9-1970. • In Debug state, it is attempting an update to the CPSR with a <fields> value that is not <fsxc>. See Behavior of MRS and MSR instructions that access the CPSR in Debug state on page C5-2097. An MSR (register) executed in User mode: • is UNPREDICTABLE if it attempts to update the SPSR • otherwise, does not update any CPSR field that is accessible only at PL1 or higher, Note MSR (register) on page A8-500 describes the valid application level uses of the MSR (register) instruction. An MSR (register) executed in System mode is UNPREDICTABLE if it attempts to update the SPSR.",
    [AOP7_B9_RFE] = "Return From Exception loads the PC and the CPSR from the word at the specified address and the following word respectively. For information about memory accesses see Memory accesses on page A8-294. RFE is: • UNDEFINED in Hyp mode. • UNPREDICTABLE in: — The cases described in Restrictions on exception return instructions on page B9-1970. Note As identified in Restrictions on exception return instructions on page B9-1970, RFE differs from other exception return instructions in that it can be executed in System mode. — Debug state.",
    [AOP7_B9_SMC_PREVIOUSLY_SMI] = "Secure Monitor Call causes a Secure Monitor Call exception. For more information see Secure Monitor Call (SMC) exception on page B1-1210. SMC is available only from software executing at PL1 or higher. It is UNDEFINED in User mode. In an implementation that includes the Virtualization Extensions: • If HCR.TSC is set to 1, execution of an SMC instruction in a Non-secure PL1 mode generates a Hyp Trap exception, regardless of the value of SCR.SCD. For more information see Trapping use of the SMC instruction on page B1-1254. • Otherwise, when SCR.SCD is set to 1, the SMC instruction is: — UNDEFINED in Non-secure state — UNPREDICTABLE if executed in a Secure PL1 mode.",
    [AOP7_B9_SRS_THUMB] = "Store Return State stores the LR and SPSR of the current mode to the stack of a specified mode. For information about memory accesses see Memory accesses on page A8-294. SRS is: • UNDEFINED in Hyp mode • UNPREDICTABLE if: — it is executed in ThumbEE state — it is executed in User or System mode — it attempts to store the Monitor mode SP when in Non-secure state — NSACR.RFR is set to 1 and it attempts to store the FIQ mode SP when in Non-secure state — it attempts to store the Hyp mode SP.",
    [AOP7_B9_SRS_ARM] = "Store Return State stores the LR and SPSR of the current mode to the stack of a specified mode. For information about memory accesses see Memory accesses on page A8-294. SRS is: • UNDEFINED in Hyp mode • UNPREDICTABLE if: — it is executed in User or System mode — it attempts to store the Monitor mode SP when in Non-secure state — NSACR.RFR is set to 1 and it attempts to store the FIQ mode SP when in Non-secure state — if it attempts to store the Hyp mode SP.",
    [AOP7_B9_STM_USER_REGISTERS] = "In a PL1 mode other than System mode, Store Multiple (user registers) stores multiple User mode registers to consecutive memory locations using an address from a base register. The processor reads the base register value normally, using the current mode to determine the correct Banked version of the register. This instruction cannot writeback to the base register. STM (User registers) is UNDEFINED in Hyp mode, and UNPREDICTABLE in User or System modes.",
    [AOP7_B9_SUBS_PC_LR_THUMB] = "The SUBS PC, LR, #<const> instruction provides an exception return without the use of the stack. It subtracts the immediate constant from LR, branches to the resulting address, and also copies the SPSR to the CPSR. Note • The instruction SUBS PC, LR, #0 is equivalent to MOVS PC, LR and ERET. • For an implementation that includes the Virtualization Extensions, ERET is the preferred disassembly of the T1 encoding defined in this section. Therefore, a disassembler might report an ERET where the original assembler code used SUBS PC, LR, #0. When executing in Hyp mode: • the encoding for SUBS PC, LR, #0 is the encoding of the ERET instruction, see ERET on page B9-1980 • SUBS PC, LR, #<const> with a nonzero constant is UNDEFINED. SUBS PC, LR, #<const> is UNPREDICTABLE: • in the cases described in Restrictions on exception return instructions on page B9-1970 • if it is executed in Debug state.",
    [AOP7_B9_SUBS_PC_LR_AND_RELATED_INSTRUCTIONS_ARM] = "The SUBS PC, LR, #<const> instruction provides an exception return without the use of the stack. It subtracts the immediate constant from LR, branches to the resulting address, and also copies the SPSR to the CPSR. The ARM instruction set contains similar instructions based on other data-processing operations, or with a wider range of operands, or both. ARM deprecates using these other instructions, except for MOVS PC, LR. All of these instructions are: • UNDEFINED in Hyp mode • UNPREDICTABLE: — in the cases described in Restrictions on exception return instructions on page B9-1970 — if executed in Debug state.",
    [AOP7_B9_VMRS] = "Move to ARM core register from Advanced SIMD and Floating-point Extension System Register moves the value of an extension system register to an ARM core register. When the specified Floating-point Extension System Register is the FPSCR, a form of the instruction transfers the FPSCR.{N, Z, C, V} condition flags to the APSR.{N, Z, C, V} condition flags. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute a VMRS instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 and Summary of access controls for Advanced SIMD functionality on page B1-1232 summarize these controls. When these settings permit the execution of floating-point and Advanced SIMD instructions, if the specified Floating-point Extension System Register is not the FPSCR, the instruction is UNDEFINED if executed in User mode. In an implementation that includes the Virtualization Extensions, when HCR.TID0 is set to 1, any VMRS access to FPSID from a Non-secure PL1 mode, that would be permitted if HCR.TID0 was set to 0, generates a Hyp Trap exception. For more information, see ID group 0, Primary device identification registers on page B1-1251. Note • VMRS on page A8-954 describes the valid application level uses of the VMRS instruction • for simplicity, the VMRS pseudocode does not show the possible trap to Hyp mode.",
    [AOP7_B9_VMSR] = "Move to Advanced SIMD and Floating-point Extension System Register from ARM core register moves the value of an ARM core register to a Floating-point system register. Depending on settings in the CPACR, NSACR, HCPTR, and FPEXC registers, and the security state and mode in which the instruction is executed, an attempt to execute a VMSR instruction might be UNDEFINED, or trapped to Hyp mode. Summary of general controls of CP10 and CP11 functionality on page B1-1230 and Summary of access controls for Advanced SIMD functionality on page B1-1232 summarize these controls. When these settings permit the execution of floating-point and Advanced SIMD instructions, if the specified Floating-point Extension System Register is not the FPSCR, the instruction is UNDEFINED if executed in User mode. Note VMSR on page A8-956 describes the valid application level uses of the VMSR instruction.",

};



#endif  /* _PLUGINS_ARM_V7_OPCODES_DESCRIPTIONS_H */
