
/* Chrysalide - Outil d'analyse de fichiers binaires
 * collection.h - prototypes pour la gestion d'éléments ajoutés par collection
 *
 * Copyright (C) 2014-2017 Cyrille Bagard
 *
 *  This file is part of Chrysalide.
 *
 *  Chrysalide is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Chrysalide is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Chrysalide.  If not, see <http://www.gnu.org/licenses/>.
 */


#ifndef _ANALYSIS_DB_COLLECTION_H
#define _ANALYSIS_DB_COLLECTION_H


#include <glib-object.h>
#include <sqlite3.h>
#include <stdbool.h>
#include <stdint.h>


#include "item.h"
#include "protocol.h"
#include "../../common/packed.h"



#define G_TYPE_DB_COLLECTION               g_db_collection_get_type()
#define G_DB_COLLECTION(obj)               (G_TYPE_CHECK_INSTANCE_CAST((obj), g_db_collection_get_type(), GDbCollection))
#define G_IS_DB_COLLECTION(obj)            (G_TYPE_CHECK_INSTANCE_TYPE((obj), g_db_collection_get_type()))
#define G_DB_COLLECTION_CLASS(klass)       (G_TYPE_CHECK_CLASS_CAST((klass), G_TYPE_DB_COLLECTION, GDbCollectionClass))
#define G_IS_DB_COLLECTION_CLASS(klass)    (G_TYPE_CHECK_CLASS_TYPE((klass), G_TYPE_DB_COLLECTION))
#define G_DB_COLLECTION_GET_CLASS(obj)     (G_TYPE_INSTANCE_GET_CLASS((obj), G_TYPE_DB_COLLECTION, GDbCollectionClass))





/* Collection générique d'éléments (instance) */
typedef struct _GDbCollection GDbCollection;

/* Collection générique d'éléments (classe) */
typedef struct _GDbCollectionClass GDbCollectionClass;


/* Indique le type défini pour une collection générique d'éléments. */
GType g_db_collection_get_type(void);

/* Prépare la mise en place d'une nouvelle collection. */
GDbCollection *g_db_collection_new(uint32_t, GType, const char *);

/* Attache à une collection un binaire pour les éléments listés. */
void g_db_collection_link_to_binary(GDbCollection *, GLoadedBinary *);

/* Décrit le type des éléments rassemblées dans une collection. */
uint32_t g_db_collection_get_feature(const GDbCollection *);

/* Décrit le type de collection manipulée. */
const char *g_db_collection_get_name(const GDbCollection *);


/* Réceptionne un élément depuis une requête réseau. */
bool _g_db_collection_unpack(GDbCollection *, packed_buffer *, DBAction *, GDbItem **);

/* Réceptionne et traite une requête réseau pour collection. */
bool g_db_collection_unpack(GDbCollection *, packed_buffer *, sqlite3 *);

/* Envoie pour traitement une requête réseau pour collection. */
bool g_db_collection_pack(GDbCollection *, packed_buffer *, DBAction, GDbItem *);

/* Envoie pour mise à jour tous les éléments courants. */
bool g_db_collection_pack_all_updates(GDbCollection *, packed_buffer *);




/* Met à disposition un encadrement des accès aux éléments. */
void g_db_collection_lock_unlock(GDbCollection *, bool, bool);

#define g_db_collection_wlock(col) g_db_collection_lock_unlock(col, true, true);
#define g_db_collection_wunlock(col) g_db_collection_lock_unlock(col, true, false);

#define g_db_collection_rlock(col) g_db_collection_lock_unlock(col, false, true);
#define g_db_collection_runlock(col) g_db_collection_lock_unlock(col, false, false);

/* Renvoie la liste des éléments rassemblés. */
GList *g_db_collection_list_items(const GDbCollection *);

/* Détermine si un élément est déjà présent ou non. */
GDbItem *g_db_collection_has_key(GDbCollection *, ...);

/* Détermine si un élément est déjà présent ou non. */
bool g_db_collection_has_item(GDbCollection *, GDbItem *);

/* Procède à l'ajout d'un nouvel élément dans la collection. */
bool _g_db_collection_add_item(GDbCollection *, GDbItem *, bool);

/* Procède au retrait d'un nouvel élément dans la collection. */
bool _g_db_collection_remove_item(GDbCollection *, GDbItem *, bool, bool);

/* Détermine l'horodatage le plus jeune pour une désactivation. */
timestamp_t _g_db_collection_compute_inactivity_timestamp(GDbCollection *, bool);

/* Met à jour le statut d'activité d'un élément de collection. */
bool _g_db_collection_update_item_activity(GDbCollection *, GDbItem *, timestamp_t *, bool);

#define g_db_collection_add_item(c, i) _g_db_collection_add_item(c, i, true)
#define g_db_collection_remove_item(c, i) _g_db_collection_remove_item(c, i, true, true)
#define g_db_collection_compute_inactivity_timestamp(c) _g_db_collection_compute_inactivity_timestamp(c, true)
#define g_db_collection_update_item_activity(c, i, t) _g_db_collection_update_item_activity(c, i, t, true)

/* Active les éléments en amont d'un horodatage donné. */
GList *g_db_collection_set_last_active(GDbCollection *, timestamp_t, timestamp_t *, sqlite3 *);

/* Désactive les éléments en aval d'un horodatage donné. */
bool g_db_collection_set_inactive(GDbCollection *, GDbItem *, timestamp_t *);



/* --------------------- MANIPULATIONS AVEC UNE BASE DE DONNEES --------------------- */


/* Crée la table d'élément dans une base de données. */
bool g_db_collection_create_db_table(const GDbCollection *, sqlite3 *);

/* Charge un ensemble d'éléments à partir d'une base de données. */
bool g_db_collection_load_all_items(GDbCollection *, sqlite3 *);



/* ------------------- CREATION DE L'ABSTRACTION POUR COLLECTIONS ------------------- */


/* Attache un binaire à une série de collections. */
void attach_binary_to_collections(GList *, GLoadedBinary *);

/* Recherche une collection correspondant à un type donné. */
GDbCollection *find_collection_in_list(GList *, uint32_t);

/* Met à disposition un encadrement des accès aux éléments. */
void lock_unlock_collections(GList *, bool, bool);

#define wlock_collections(lst) lock_unlock_collections(lst, true, true);
#define wunlock_collections(lst) lock_unlock_collections(lst, true, false);

#define rlock_collections(lst) lock_unlock_collections(lst, false, true);
#define runlock_collections(lst) lock_unlock_collections(lst, false, false);

/* Collecte les informations utiles pour un nouvel arrivant. */
bool pack_all_collection_updates(GList *, packed_buffer *);

/* Met à jour les statuts d'activité des éléments. */
bool update_activity_in_collections(GList *, packed_buffer *, sqlite3 *);



#endif  /* _ANALYSIS_DB_COLLECTION_H */
