
/* Chrysalide - Outil d'analyse de fichiers binaires
 * protocol.h - prototypes pour la description du protocole impactant les BD Chrysalide
 *
 * Copyright (C) 2014-2017 Cyrille Bagard
 *
 *  This file is part of Chrysalide.
 *
 *  Chrysalide is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Chrysalide is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Chrysalide.  If not, see <http://www.gnu.org/licenses/>.
 */


#ifndef _ANALYSIS_DB_PROTOCOL_H
#define _ANALYSIS_DB_PROTOCOL_H



/**
 * Version de la définition courante du protocole.
 */
#define CDB_PROTOCOL_VERSION 0xc0de0003




/**
 * Délai maximal de réaction pour les coupures de flux (en ms).
 */
#define CDB_CONN_TIMEOUT 1000





/* Comportement vis à vis des éléments */
typedef enum _DBStorage
{
    DBS_ALL_LOCAL           = 0x00,         /* Enregistrements locaux      */
    DBS_ALL_REMOTE          = 0x01,         /* Enregistrements distants    */

    DBS_MAX = 0x01

} DBStorage;












/**
 * Une fois la connexion établie, les paquets ont tous la forme suivante :





 *
 *    [ type de collection visée ; cf. DBFeatures ]
 *    [ action à mener ; cf. DBAction             ]
 *    [ élément de type GDbItem sérialisé...      ]
 *
 */

/* Fonctionnalités offertes nativement */
typedef enum _DBFeatures
{
    DBF_BOOKMARKS,                          /* Signets dans le code        */
    DBF_COMMENTS,                           /* Commentaires ajoutés        */
    DBF_MOVES,                              /* Déplacements dans du code   */
    DBF_DISPLAY_SWITCHERS,                  /* Choix d'affichage           */

    DBF_COUNT

} DBFeatures;

/* Interactions disponibles vis à vis d'une collection. */
typedef enum _DBAction
{
    DBA_ADD_ITEM,                           /* Ajout d'un élément          */
    DBA_REM_ITEM,                           /* Suppression d'un élément    */

    DBA_CHANGE_STATE,                       /* Changement d'activité       */

    DBA_COUNT

} DBAction;






/**
 * Commandes envoyées d'un côté à un autre.
 */
typedef enum _DBCommand
{
    DBC_HELO,                               /* Connexion initiale C -> S   */
    DBC_WELCOME,                            /* Réponse initiale S -> C     */
    DBC_SAVE,                               /* Enregistrement de l'archive */
    DBC_COLLECTION,                         /* Implication d'une collection*/

    /**
     * Gestion de la commande 'DBC_[GS]ET_ALL_ITEMS'.
     *
     * Un client qui se connecte à un serveur doit en premier lieu envoyer :
     *
     *    [ Demande de mise à jour : DBC_GET_ALL_ITEMS  ]
     *
     * Tant qu'il ne reçoit pas la commande DBC_SET_ALL_ITEMS depuis le
     * serveur, toutes les actions sur une collection sont à rejeter car elles
     * lui seront retransmises plus tard.
     *
     * De son côté, le serveur répond par une requête :
     *
     *    [ Notification de maj : DBC_SET_ALL_ITEMS     ]
     *
     * Dans la foulée, il enverra ensuite les éléments avec des paquets classiques :
     *
     *    [ Traitement de collection : DBC_COLLECTION   ]
     *    [ Action : DBA_ADD_ITEM                       ]
     *    ...
     *
     * Les traitements se réalisent dans :
     *  - g_db_client_update() pour la partie client.
     *  - g_cdb_archive_process() pour la partie serveur.
     *
     */

    DBC_GET_ALL_ITEMS,                      /* Mise à jour à la connexion  */
    DBC_SET_ALL_ITEMS,                      /* Mise à jour à la connexion  */

    /**
     * Gestion de la commande 'DBC_SET_LAST_ACTIVE'.
     *
     * Le client connecté envoie un paquet de la forme suivante :
     *
     *    [ Statut d'historique : DBC_SET_LAST_ACTIVE   ]
     *    [ <horodatage du dernier élément actif        ]
     *
     * Le serveur s'exécute et notifie le client d'éventuels changements,
     * avec une série de paquets de la forme :
     *
     *    [ Traitement de collection : DBC_COLLECTION   ]
     *    [ Action : DBC_SET_LAST_ACTIVE                ]
     *    [ <élément dont le statut a évolué>           ]
     *
     * Les traitements se réalisent dans :
     *  - g_db_collection_set_last_active() pour la partie serveur.
     *  - g_db_client_set_last_active() pour la partie client.
     *
     */

    DBC_SET_LAST_ACTIVE,                    /* Définition du dernier actif */

    DBC_COUNT

} DBCommand;







/**
 * Erreurs pouvant survenir...
 */
typedef enum _DBError
{
    DBE_NONE,                               /* Succès d'une opération      */
    DBE_BAD_EXCHANGE,                       /* Incohérence des échanges    */

    DBE_WRONG_VERSION,                      /* Proto Client != Serveur     */
    DBE_SYS_ERROR,                          /* Erreur suite à un appel sys.*/
    DBE_ARCHIVE_ERROR,                      /* Soucis du côté libarchive   */
    DBE_XML_VERSION_ERROR,                  /* Vieille archive présente    */
    DBE_DB_LOADING_ERROR,                   /* Erreur pendant le chargement*/

    DBE_COUNT

} DBError;





/**
 * Gestion de la commande 'DBC_SAVE'.
 *
 * Le client connecté envoie un paquet de la forme suivante :
 *
 *    [ Ordre de sauvegarde : DBC_SAVE            ]
 *
 * Le serveur s'exécute et renvoie un bilan :
 *
 *    [ Ordre de sauvegarde : DBC_SAVE            ]
 *    [ Statut d'exécution ; cf. DBError          ]
 *
 * Les traitements se réalisent dans :
 *  - g_db_client_save() pour la partie client en émission.
 *  - g_cdb_archive_process() pour la partie serveur.
 *  - g_db_client_update() pour la partie client en réception.
 *
 */








#endif  /* _ANALYSIS_DB_PROTOCOL_H */
