
/* Chrysalide - Outil d'analyse de fichiers binaires
 * format-int.h - prototypes utiles aux formats binaires
 *
 * Copyright (C) 2009-2017 Cyrille Bagard
 *
 *  This file is part of Chrysalide.
 *
 *  Chrysalide is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Chrysalide is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Chrysalide.  If not, see <http://www.gnu.org/licenses/>.
 */


#ifndef _FORMAT_FORMAT_INT_H
#define _FORMAT_FORMAT_INT_H


#include "format.h"


#include "preload.h"
#include "../gtkext/gtkstatusstack.h"
#include "../mangling/demangler.h"



/* ------------------------ TRAITEMENT INDIVIDUEL DE FORMATS ------------------------ */


/* Indique la désignation interne du format. */
typedef const char * (* format_get_name_fc) (const GBinFormat *);

/* Fournit une description humaine du format. */
typedef const char * (* format_get_desc_fc) (const GBinFormat *);

/*Assure l'interprétation d'un format en différé. */
typedef bool (* format_analyze_fc) (GBinFormat *, wgroup_id_t, GtkStatusStack *);

/* Indique le boutisme employé par le format binaire analysé. */
typedef SourceEndian (* format_get_endian_fc) (const GBinFormat *);

/* Réalise un traitement post-désassemblage. */
typedef void (* format_complete_analysis_fc) (GBinFormat *, wgroup_id_t, GtkStatusStack *);


/* Rythme des allocations pour les entrées de code */
#define EXTRA_POINT_BLOCK 100


/* Description d'une erreur */
typedef struct _fmt_error
{
    BinaryFormatError type;                 /* Type d'erreur               */

    vmpa2t addr;                            /* Localisation du problème    */
    char *desc;                             /* Description du soucis       */

} fmt_error;


/* Format binaire générique (instance) */
struct _GBinFormat
{
    GObject parent;                         /* A laisser en premier        */

    GBinContent *content;                   /* Contenu binaire à étudier   */

    virt_t *entry_points;                   /* Points d'entrée du code     */
    size_t ep_count;                        /* Nombre de ces points        */

    virt_t *extra_points;                   /* Autres débuts de code       */
    size_t xp_allocated;                    /* Taille d'inscription allouée*/
    size_t xp_count;                        /* Nombre de points enregistrés*/

    GRWLock pt_lock;                        /* Accès à la liste des points */

    GPreloadInfo *info;                     /* Préchargements du format    */

    GCompDemangler *demangler;              /* Décodage de noms privilégié */

    GBinSymbol **symbols;                   /* Liste des symboles trouvés  */
    size_t sym_count;                       /* Quantité de ces symboles    */
    unsigned int sym_stamp;                 /* Marque de suivi des modifs  */
    GRWLock syms_lock;                      /* Accès à la liste de symboles*/
#ifndef NDEBUG
    gint sym_locked;                        /* Statut d'accès à la liste   */
#endif

    fmt_error *errors;                      /* Liste d'erreurs rencontrées */
    size_t error_count;                     /* Taille de cette liste       */
    GMutex error_mutex;                     /* Verrou pour l'accès         */
#ifndef NDEBUG
    gint error_locked;                      /* Statut d'accès à la liste   */
#endif

};

/* Format binaire générique (classe) */
struct _GBinFormatClass
{
    GObjectClass parent;                    /* A laisser en premier        */

    format_get_name_fc get_name;            /* Désignation interne         */
    format_get_desc_fc get_desc;            /* Désignation humaine         */

    format_analyze_fc analyze;              /* Interprétation du format    */

    format_get_endian_fc get_endian;        /* Boutisme employé            */

    format_complete_analysis_fc complete;   /* Terminaison d'analyse       */

    /* Signaux */

    void (* symbol_added) (GBinFormat *, GBinSymbol *);
    void (* symbol_removed) (GBinFormat *, GBinSymbol *);

};


/* Effectue les ultimes opérations de chargement d'un binaire. */
bool g_binary_format_complete_loading(GBinFormat *, GtkStatusStack *);

/* Définit le contenu binaire à analyser. */
void g_binary_format_set_content(GBinFormat *, GBinContent *);




/* ------------------------------ DECODAGE DE SYMBOLES ------------------------------ */


/* Décode une chaîne de caractères donnée en type. */
GDataType *g_binary_format_decode_type(const GBinFormat *, const char *);

/* Décode une chaîne de caractères donnée en routine. */
GBinRoutine *g_binary_format_decode_routine(const GBinFormat *, const char *);



#endif  /* _FORMAT_FORMAT_INT_H */
