
/* Chrysalide - Outil d'analyse de fichiers binaires
 * gbuffercache.h - prototypes pour l'affichage à la demande d'un ensemble de lignes
 *
 * Copyright (C) 2016-2017 Cyrille Bagard
 *
 *  This file is part of Chrysalide.
 *
 *  Chrysalide is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Chrysalide is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Chrysalide.  If not, see <http://www.gnu.org/licenses/>.
 */


#ifndef _GLIBEXT_GBUFFERCACHE_H
#define _GLIBEXT_GBUFFERCACHE_H


#include <glib-object.h>
#include <stdbool.h>
#include <gdk/gdk.h>


#include "gdisplayoptions.h"
#include "gwidthtracker.h"
#include "linegen.h"



/* -------------------------- TAMPON POUR CODE DESASSEMBLE -------------------------- */


#define G_TYPE_BUFFER_CACHE             (g_buffer_cache_get_type())
#define G_BUFFER_CACHE(obj)             (G_TYPE_CHECK_INSTANCE_CAST((obj), G_TYPE_CODE_BUFFER, GBufferCache))
#define G_BUFFER_CACHE_CLASS(klass)     (G_TYPE_CHECK_CLASS_CAST((klass), G_TYPE_CODE_BUFFER, GBufferCacheClass))
#define G_IS_BUFFER_CACHE(obj)          (G_TYPE_CHECK_INSTANCE_TYPE((obj), G_TYPE_CODE_BUFFER))
#define G_IS_BUFFER_CACHE_CLASS(klass)  (G_TYPE_CHECK_CLASS_TYPE((klass), G_TYPE_CODE_BUFFER))
#define G_BUFFER_CACHE_GET_CLASS(obj)   (G_TYPE_INSTANCE_GET_CLASS((obj), G_TYPE_CODE_BUFFER, GBufferCacheClass))


/* Tampon pour gestion de lignes optimisée (instance) */
typedef struct _GBufferCache GBufferCache;

/* Tampon pour gestion de lignes optimisée (classe) */
typedef struct _GBufferCacheClass GBufferCacheClass;


/* Détermine le type du composant de tampon pour gestion de lignes optimisée. */
GType g_buffer_cache_get_type(void);

/* Crée un nouveau composant de tampon pour code désassemblé. */
GBufferCache *g_buffer_cache_new(const GBinContent *);

/* Fournit la hauteur d'impression d'une ligne visualisée. */
gint g_buffer_cache_get_line_height(const GBufferCache *);

/* Fournit la taille réservée pour la marge gauche. */
gint g_buffer_cache_get_left_margin(const GBufferCache *);

/* Fournit la position de départ pour l'impression de texte. */
gint g_buffer_cache_get_text_position(const GBufferCache *);

/* Compte le nombre de lignes rassemblées dans un tampon. */
size_t g_buffer_cache_count_lines(const GBufferCache *);

/* Fournit un lien vers la structure de suivi de largeurs. */
GWidthTracker *g_buffer_cache_get_width_tracker(const GBufferCache *);

/* Insère un générateur dans des lignes à une position donnée. */
void g_buffer_cache_insert_at(GBufferCache *, size_t, GLineGenerator *, BufferLineFlags, bool, bool);

/* Retire une ligne du tampon. */
void g_buffer_cache_delete_at(GBufferCache *, size_t);

/* Retire un type de générateur de lignes. */
GLineGenerator *g_buffer_cache_delete_type_at(GBufferCache *, size_t, GType, bool, bool);

/* Ajoute en fin de tampon un générateur de lignes. */
void g_buffer_cache_append(GBufferCache *, GLineGenerator *, BufferLineFlags);

/* Etend un tampon avec un générateur de lignes unique. */
void g_buffer_cache_extend_with(GBufferCache *, size_t, GLineGenerator *);

/* Réduit le tampon à une quantité de lignes précise. */
void g_buffer_cache_truncate(GBufferCache *, size_t);

/* Retrouve l'emplacement correspondant à une position de ligne. */
void g_buffer_cache_get_line_cursor(const GBufferCache *, size_t, gint, GLineCursor **);

/* Détermine l'ensemble des propriétés attachées à une ligne. */
BufferLineFlags g_buffer_cache_get_line_flags(const GBufferCache *, size_t);

/* Retrouve une ligne au sein d'un tampon avec un indice. */
GBufferLine *g_buffer_cache_find_line_by_index(const GBufferCache *, size_t);

/* Fait remonter les largeurs requises par une ligne donnée. */
void g_buffer_cache_collect_widths(GBufferCache *, size_t, line_width_summary *);

/* Imprime une partie choisie du tampon contenant des lignes. */
void g_buffer_cache_draw(const GBufferCache *, cairo_t *, size_t, size_t, const cairo_rectangle_int_t *, const GDisplayOptions *, const line_width_summary *, const gint *, const segcnt_list *);

/* Indique l'indice correspondant à une adresse donnée. */
size_t _g_buffer_cache_find_index_by_cursor(const GBufferCache *, const GLineCursor *, bool, size_t, size_t);

/* Indique l'indice correspondant à une adresse donnée. */
size_t g_buffer_cache_find_index_by_cursor(const GBufferCache *, const GLineCursor *, bool);

/* Avance autant que possible vers une ligne idéale. */
size_t g_buffer_cache_look_for_flag(const GBufferCache *, size_t, BufferLineFlags);

/* Indique la position d'affichage d'une adresse donnée. */
bool g_buffer_cache_get_cursor_coordinates(const GBufferCache *, const GLineCursor *, size_t, size_t, bool, gint *, gint *);



#endif  /* _GLIBEXT_GBUFFERCACHE_H */
